// vector.cpp
//
// Source code for (double) vectors
//
// Jiangguo (James) Liu, ColoState, 01/2007--10/2011
// With the great help of Rachel Cali (Spring 2007)


#include <cmath>
#include <cstdlib>
#include <iostream>
#include "vector.h"
using namespace std;


// Vector: Constructor with given size and all elements a or 0

Vector::Vector(int numElts, double a)
{
   n = numElts;
   p = new double[n];
   for (int i=0; i<n; ++i)  p[i] = a;
}


// Vector: constructor with given size and data

Vector::Vector(int numElts, double *data)
{
   n = numElts;
   p = new double[n];
   for (int i=0; i<n; ++i)  p[i] = data[i];
}


// Vector: copy constructor

Vector::Vector(const Vector &v) 
{
   n = v.n;
   p = new double[n];
   for (int i=0; i<n; ++i)  p[i] = v.p[i]; 
}


// Vector: copy assignment

Vector &Vector::operator=(const Vector &v) 
{
   if (this != &v) {
      delete[] p;
      n = v.n;
      p = new double[n];
      for (int i=0; i<n; ++i)  p[i] = v.p[i];
   }
   return *this;
}


// Vector: resize the vector

void Vector::resize(int numElts) 
{
   delete[] p;
   n = numElts;
   p = new double[n];
   for (int i=0; i<n; ++i)  p[i] = 0.0;
   return;
}


// Vector: l_infty norm

double Vector::l0norm() const
{
   double norm = fabs(p[0]);
   for (int i=1; i<n; ++i) {
      if (norm<fabs(p[i]))  norm = fabs(p[i]);
   }
   return norm;
}


// Vector: l_1 norm

double Vector::l1norm() const
{
   double norm = 0.0;
   for (int i=0; i<n; ++i)  norm += fabs(p[i]);
   return norm;
}


// Vector: l_2 norm

double Vector::l2norm() const
{
   double norm = 0.0;
   for (int i=0; i<n; ++i) {
      double tmp = fabs(p[i]);
      norm += tmp*tmp;
   }
   return sqrt(norm);
}

// Vector: get subvector


Vector Vector::get(int i, int length)
{
   if (n < i+length){
      cout << "Bad vector sizes\n";
      exit(EXIT_FAILURE);
   }
   Vector v(length,0.0);
   for (int j=0; j<length; ++j)
      v.p[j] = p[i+j];
   return v; 
}


// Vector: replace subvector

Vector Vector::replace(int i, const Vector &v)
{
   if (n < i+v.size()){
      cout << "Bad vector sizes\n";
      exit(EXIT_FAILURE);
   }
   for (int j=0; j<v.size(); ++j)
      p[i+j] = v.p[j];
   return *this;
} 


// Vector: save to a data file 

void Vector::save2file(char fn[20]) const 
{
   FILE *fp;

   if (NULL==(fp=fopen(fn, "w"))) {
      puts("Open data file failed.  Exit!");
      exit(-1);
   }

   fprintf(fp, "%8d\n", n);

   for (int i=0; i<n; ++i) 
      fprintf(fp, "%8d  %21.9f\n", i+1, p[i]);

   fclose(fp);
   return;
}


// Vector: v += w

Vector &Vector::operator+=(const Vector &w) 
{
   if (n != w.n) {
      cout << "Bad vector sizes\n";
      exit(EXIT_FAILURE);
   }
   for (int i=0; i<n; ++i)  p[i] += w.p[i];
   return *this;
}


// Vector: v -= w

Vector &Vector::operator-=(const Vector &w) 
{
   if (n != w.n) {
      cout << "Bad vector sizes\n";
      exit(EXIT_FAILURE);
   }
   for (int i=0; i<n; ++i)  p[i] -= w.p[i];
   return *this;
}


// Vector: Unary -

Vector Vector::operator-() const
{
   Vector w(n, 0.0);
   for (int i=0; i<n; ++i)  w[i] = -p[i];
   return w;
}


// Vector: v = v1 + v2

Vector operator+(const Vector &v1, const Vector &v2) 
{
   if (v1.n != v2.n)  {
      cout << "Bad vector sizes!\n";
      exit(EXIT_FAILURE);
   }
   Vector v(v1);
   v += v2;
   return v;
}


// Vector: v = v1 - v2
// Notice that the implementation is different than 
// that in the overloaded binary operator +

Vector operator-(const Vector &v1, const Vector &v2) 
{
   if (v1.n != v2.n) {
      cout << "Bad vector sizes!\n";
      exit(EXIT_FAILURE);
   }
   Vector v(v1.n, 0.0);
   for (int i=0; i<v1.n; ++i)  v[i] = v1[i] - v2[i];
   return v;
}


// Vector: v = v1*v2, componentwise
/*
Vector operator*(const Vector &v1, const Vector &v2) 
{
   if (v1.n != v2.n) {
      cout << "Bad vector sizes!\n";
      exit(EXIT_FAILURE);
   }
   Vector v(v1.n, 0.0);
   for (int i=0; i<v1.n; ++i)  v[i] = v1[i]*v2[i];
   return v;
}
*/


// Vector: v = v1/v2, componentwise
/*
Vector operator/(const Vector &v1, const Vector &v2) 
{
   if (v1.n != v2.n) {
      cout << "Bad vector sizes!\n";
      exit(EXIT_FAILURE);
   }
   Vector v(v1.n, 0.0);
   for (int i=0; i<v1.n; ++i) {
      if (v2[i]==0)  {
         cout << "Divided by zero!\n";
         exit(EXIT_FAILURE);
      }
      v[i] = v1[i]/v2[i];
   }
   return v;
}
*/


// Vector: scalar-vector multiplication

Vector operator*(const double a, const Vector &v)
{
   Vector u(v.n, 0.0);
   for (int i=0; i<v.n; ++i)  u[i] = a*v[i];
   return u;
}


//  Vector: dot product

double operator*(const Vector &v1, const Vector &v2)
{
   if (v1.n != v2.n) {
      cout << "Bad vector sizes!\n";
      exit(EXIT_FAILURE);
   }
   double dp = 0.0;
   for (int i=0; i<v1.n; ++i)  dp += v1[i]*v2[i];
   return dp;
}


// Vector: dot product

double dotProd(const Vector &v1, const Vector &v2)
{
   if (v1.n != v2.n) {
      cout << "Bad vector sizes!\n";
      exit(EXIT_FAILURE);
   }
   double dp = 0.0;
   for (int i=0; i<v1.n; ++i)  dp += v1[i]*v2[i];
   return dp;
}


// Vector: output stream

std::ostream &operator<<(std::ostream &strm, const Vector &v) 
{
   for (int i=0; i<v.n; i++) {
      strm << v[i] << "  ";
      if (i%5==4)  strm<<"\n";
   }
   return strm;
}
