// matrix.h
//
// Classes for (double) full and sparse matrices
//
// Jiangguo (James) Liu, ColoState, 01/2007--10/2011 
// With the great help of Rachel Cali (Spring 2007)


#ifndef MATRIX_H
#define MATRIX_H

#include <iostream>
#include "vector.h"


// Full small size matrices

class FullMatrix {
public: 
   FullMatrix() {m=0; n=0; p=0;}  // Default constructor
   FullMatrix(int numRows, int numCols);  // Another constructor
   FullMatrix(const FullMatrix &);  // Copy constructor
   FullMatrix &operator=(const FullMatrix &);  // Copy assignment
   ~FullMatrix() {delete[] p; p=0; m=0; n=0;}  // Destructor
   void reshape(int numRows, int numCols);
   int rowSize() const {return m;}
   int colSize() const {return n;}
   double &operator()(int i, int j) const;  // 1<=i<=m, 1<=j<=n
   void save2file(char fn[20]) const;  // save to a data file 
friend std::ostream &operator<<(std::ostream &strm, const FullMatrix&);
friend FullMatrix transpose(const FullMatrix&);  // Transpose
friend FullMatrix operator+(const FullMatrix&, const FullMatrix&);
   // Matrix addition
friend FullMatrix operator*(const double, const FullMatrix&);
   // Scalar-matrix multiplication
friend Vector operator*(const FullMatrix&, const Vector&);
friend double dotProd(const FullMatrix&, const FullMatrix&);  // Dot product
private: 
   int m, n;
   double *p;
};


// Diagonal matrices
// mainly for preconditioning in solving large sparse linear systems

class DiagMatrix {
public: 
   DiagMatrix() {n=0; p=0;}
   DiagMatrix(int order);
   DiagMatrix(const DiagMatrix &);
   DiagMatrix &operator=(const DiagMatrix &);
   ~DiagMatrix() {delete[] p; p=0; n=0;}
   void reshape(int order);
   int size() const {return n;}
   double getEntry(int i) const {return p[i-1];}
   void setEntry(int i, double a) {p[i-1]=a;}
   void addEntry(int i, double a) {p[i-1]+=a;}
   double det() const;
   void save2file(char fn[20]) const;
private: 
   int n;
   double *p;
};


// Sparse matrices in the Compressed Row Storage (CRS) format

class SparseMatrix {
public: 
   SparseMatrix() {m=0; n=0; rowoff=0; colind=0; val=0;}
   SparseMatrix(int numRows, int numCols, int numNonZeros);
   SparseMatrix(const SparseMatrix &);
   SparseMatrix &operator=(const SparseMatrix &);
   ~SparseMatrix() {delete[] rowoff,colind,val; m=0; n=0; nnz=0;}
   void reshape(int numRows, int numCols, int *numEntrsInRow, int **colIdx);
   int getRowSize() const {return m;}
   int getColSize() const {return n;}
   int getNumNonZeros() const {return nnz;}
   double getEntry(int i, int j) const;
   void setEntry(int i, int j, double a);
   void addEntry(int i, int j, double a);
   void zeroutRow(int i);
   void save2file(char fn[20]) const;
friend SparseMatrix operator+(const SparseMatrix&, const SparseMatrix&);
friend SparseMatrix operator*(double, const SparseMatrix&);
friend Vector operator*(const SparseMatrix&, const Vector&);
friend DiagMatrix diagonal(const SparseMatrix&);
private: 
   int m, n;  // Matrix row,column sizes
   int nnz;  // Total number of nonzero entries
   int *rowoff;  // row off(set), length=m
   int *colind;  // col(umn) ind(ex), length=nnz
   double *val;  // val(ue), length=nnz
};


// Sparse block matrices in the Block Compressed Row Storage (BCRS) format
// Block structure convenient for FE assembly, etc.
// Sparse structure also

class SparseBlockMatrix {
public: 
   SparseBlockMatrix();
// SparseBlockMatrix(int numRowBlks, int numColBlks, int numNzBlks);
   SparseBlockMatrix(int numRowBlks, int numColBlks, int numNzBlks, 
      int *dimRowBlk, int *dimColBlk, int *blkRowOff, int *blkColInd);
   ~SparseBlockMatrix();
   int numRowBlks() const {return nrb;}
   int numColBlks() const {return ncb;}
   int numNzBlks() const {return nnzb;}
   int rowSize() const {return m;}
   int colSize() const {return n;}
   int numNonZeros() const {return nnz;}
   void reset(int numRowBlks, int numColBlks, int numNzBlks, 
      int *dimRowBlk, int *dimColBlk, int *blkRowOff, int *blkColInd);
   void add(int ib, int jb, const FullMatrix &A);
   void replace(int ib, int jb, const FullMatrix &A);
   FullMatrix get(int ib, int jb) const;
friend std::ostream &operator<<(std::ostream &strm, const SparseBlockMatrix&);
friend Vector operator*(const SparseBlockMatrix&, const Vector&);
private: 
  int nrb, ncb, nnzb;  // Number of row/column/nonzero blocks
  int *drb, *dcb;  // Dim of row/column blocks, length = nrb,ncb, respctively
  int *brb, *bcb;  // Beginning row/column blocks, length = nrb,ncb, resp.
  int *bro, *bci;  // block rowoff/colind, length = nrb+1,nnzb, respectively
  FullMatrix *bv;
  int m, n;  // Matrix row, column sizes
  int nnz;  // Total number of nonzero entries
/*
  int *rowoff;  // row off(set), length=m
  int *colind;  // col(umn) ind(ex), length=nnz
  double *val;  // val(ue), length=nnz
*/
};


// Block diagonal matrices

class BlockDiagMatrix {
public: 
   BlockDiagMatrix() {nb=0; bb=0; db=0; bv=0; n=0;}  // Default constructor
   BlockDiagMatrix(const SparseBlockMatrix&);
      // A constructor based on a sparse block matrix
   ~BlockDiagMatrix() {delete[] bb,db,bv; nb=0; n=0;}  // Destructor 
   int numBlks() const {return nb;}
   int bgnBlk(int ib) const {return bb[ib];}
   int dimBlk(int ib) const {return db[ib];}
   int size() const {return n;}
   FullMatrix get(int ib) const {return bv[ib];} 
      // Get the ib-th diagonal block
private: 
   int nb;  // Number of blocks
   int *bb; // Begin block
   int *db;  // Dimension of block
   FullMatrix *bv;
   int n;  // The order of the matrix
};


#endif  // MATRIX_H
