// matrix.cpp
//
// Source code for (double) full, sparse, and sparse block matrices
//
// Jiangguo (James) Liu, ColoState, 01/2007--10/2011
// With the great help of Rachel Cali (Spring 2007)


#include <cmath>
#include <cstdlib>
#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include "matrix.h"
#include "vector.h"
using namespace std;


// FullMatrix: another constructor

FullMatrix::FullMatrix(int numRows, int numCols) 
{
   m = numRows;
   n = numCols;
   p = new double[m*n];
   for (int k=0; k<m*n; ++k)  p[k]=0.0;
}


// FullMatrix: copy constructor

FullMatrix::FullMatrix(const FullMatrix &A) 
{
   m = A.m;
   n = A.n;
   p = new double[m*n];
   for (int k=0; k<m*n; ++k)  p[k]=A.p[k];
}


// FullMatrix: copy assignment

FullMatrix &FullMatrix::operator=(const FullMatrix &A) 
{
   if (this != &A) {
      delete[] p;
      m = A.m;
      n = A.n;
      p = new double[m*n];
      for (int k=0; k<m*n; ++k)  p[k] = A.p[k];
   }
   return *this;
}


// FullMatrix: reshape
// almost like a constructor

void FullMatrix::reshape(int numRows, int numCols) 
{
   delete[] p;
   m = numRows;
   n = numCols;
   p = new double[m*n];
   for (int k=0; k<m*n; ++k)  p[k] = 0.0;
   return;
}


// FullMatrix: mathmatical style access 

double &FullMatrix::operator()(int i, int j) const 
{
   return p[(i-1)*n+(j-1)];  // Row-wise storage 
}


// FullMatrix: save to a data file 

void FullMatrix::save2file(char fn[20]) const 
{
   FILE *fp;

   if (NULL==(fp=fopen(fn, "w"))) {
      puts("Open data file failed.  Exit.");
      exit(-1);
   }

   fprintf(fp, "%8d  %8d\n", m, n);

   for (int i=1; i<=m; ++i) 
      for (int j=1; j<=n; ++j) 
         fprintf(fp, "%+18.12f\n", p[(i-1)*n+(j-1)]);
         // fprintf(fp, "%8d  %8d  %+18.12f\n", i, j, p[(i-1)*n+(j-1)]);
   fclose(fp);
   return;
}


// FullMatrix: cout for a small size matrix

std::ostream &operator<<(std::ostream &strm, const FullMatrix &A) 
{
   strm << A.m << "   " << A.n << "\n";
   for (int i=1; i<=A.m; ++i) {
      for (int j=1; j<=A.n; ++j) {
         // strm << A(i,j) << "   ";
         printf("%+9.6G  ", A(i,j));
      }
      // cout << "\n";
      printf("\n");
   }
   return strm;
}


// FullMatrix: transpose 

FullMatrix transpose(const FullMatrix &A) 
{
   FullMatrix B(A.n, A.m);

   for (int i=1; i<=A.m; ++i) 
      for (int j=1; j<=A.n; ++j) 
         B(j,i) = A(i,j);

   return B;
}


// Matrix addition

FullMatrix operator+(const FullMatrix &A, const FullMatrix &B) 
{
   FullMatrix C(A.m, A.n);

   if (A.m!=B.m || A.n!=B.n) {
      cout << "Bad matrix sizes!\n";
      exit(EXIT_FAILURE);
   }

   for (int i=1; i<=A.m; ++i) 
      for (int j=1; j<=A.n; ++j) 
         C(i,j) = A(i,j) + B(i,j);

   return C;
}


// Scalar-vector multiplication

FullMatrix operator*(const double a, const FullMatrix &A) 
{
   FullMatrix B(A.m, A.n);

   for (int i=1; i<=A.m; ++i) 
      for (int j=1; j<=A.n; ++j) 
         B(i,j) = a*A(i,j);

   return B;
}


// * overloaded: full-matrix-vector multiplication

Vector operator*(const FullMatrix&A, const Vector&v) 
{
   if (A.n!=v.size()) {
      cout << "Matrix vector sizes do not match!\n";
      exit(EXIT_FAILURE);
   }

   Vector u(A.m);

   for (int i=1; i<=A.m; ++i) 
      for (int j=1; j<=A.n; ++j) 
         u(i) += A(i,j)*v(j);

   return u;
}


// FullMatrix: dot product
// finished by RC

double dotProd(const FullMatrix &A, const FullMatrix &B) 
{
   if (A.m!=B.m || A.n!=B.n) {
      cout << "Matrix sizes do not match!\n";
      exit(EXIT_FAILURE);
   }
   
   double dp = 0.0;

   for (int i=1; i<=A.m; ++i) 
      for (int j=1; j<=A.n; ++j) 
         dp += A(i,j)*B(i,j);

   return dp;
}


// DiagMatrix: a constructor

DiagMatrix::DiagMatrix(int order)
{
   n = order;
   p = new double[n];
   for (int i=0; i<n; ++i)  p[i] = 0.0;
}


// DiagMatrix: copy constructor

DiagMatrix::DiagMatrix(const DiagMatrix &A) 
{
   n = A.n;
   p = new double[n];
   for (int i=0; i<n; ++i)  p[i] = A.p[i];
}


// DiagMatrix: copy assignment

DiagMatrix &DiagMatrix::operator=(const DiagMatrix &A) 
{
   if (this != &A) {
      delete[] p;
      n = A.n;
      p = new double[n];
      for (int i=0; i<n; ++i)  p[i] = A.p[i];
   }
   return *this;
}


// DiagMatrix: reshape, almost like a constructor

void DiagMatrix::reshape(int order) 
{
   delete[] p;
   n = order;
   p = new double[n];
   for (int i=0; i<n; ++i)  p[i] = 0.0;
   return;
}


// DiagMatrix: determinant

double DiagMatrix::det() const 
{
   double d = 1;
   for (int i=0; i<n; ++i)  d *= p[i];
   return d;
}


// DiagMatrix: save to a data file 

void DiagMatrix::save2file(char fn[20]) const 
{
   FILE *fp;

   if (NULL==(fp=fopen(fn, "w"))) {
      puts("Open data file failed.  Exit!");
      exit(-1);
   }

   fprintf(fp, "%8d\n", n);

   for (int i=0; i<n; ++i) 
      fprintf(fp, "%8d  %21.9f\n", i+1, p[i]);

   fclose(fp);
   return;
}


// SparseMatrix: a constructor with not much info

SparseMatrix::SparseMatrix(int numRows, int numCols, int numNonZeros) 
{
   m = numRows;
   n = numCols;
   nnz = numNonZeros;

   rowoff = new int[m+1];
   colind = new int[nnz];
   val = new double[nnz];

   for (int k=0; k<nnz; ++k)  val[k] = 0.0;
}


// SparseMatrix: copy constructor

SparseMatrix::SparseMatrix(const SparseMatrix &A) 
{
   m = A.m;  n = A.n;  nnz= A.nnz;

   rowoff = new int[m+1];
   colind = new int[nnz];
   val = new double[nnz];

   for (int i=0; i<=m; ++i)  rowoff[i] = A.rowoff[i];

   for (int k=0; k<nnz; ++k) {
      colind[k] = A.colind[k];
      val[k] = A.val[k];
   }
}


// SparseMatrix: copy assignment

SparseMatrix &SparseMatrix::operator=(const SparseMatrix &A) 
{
   if (this != &A) {
      delete[] rowoff, colind, val;

      m = A.m;  n = A.n;  nnz= A.nnz;

      rowoff = new int[m+1];
      colind = new int[nnz];
      val = new double[nnz];

      for (int i=0; i<=m; ++i)  rowoff[i] = A.rowoff[i];

      for (int k=0; k<nnz; ++k) {
         colind[k] = A.colind[k];
         val[k] = A.val[k];
      }
   }

   return *this;
}


// SparseMatrix: reshape
// almost like a constructor

void SparseMatrix::reshape(int numRows, int numCols, 
  int *numEntrsInRow, int **colIdx) 
{
  int i;
   delete[] rowoff,colind,val;

   m = numRows;
   n = numCols;
   nnz = 0;
   for (i=0; i<m; ++i)  nnz += numEntrsInRow[i];
 
   rowoff = new int[m+1];
   colind = new int[nnz];
   val = new double[nnz];

   int k = 0;
   for (i=0; i<m; ++i) {
      rowoff[i] = k+1;
      for (int j=0; j<numEntrsInRow[i]; ++j) {
         colind[k] = colIdx[i][j];
         val[k] = 0.0;
         k++;
      }
   }
   rowoff[m] = nnz+1;

   return;
}


// SparseMatrix: get entry (i,j), 1<=i<=m, 1<=j<=n

double SparseMatrix::getEntry(int i, int j) const 
{
   double a=0.0;
   for (int k=rowoff[i-1]; k<rowoff[i]; ++k) {
      if (j==colind[k-1]) {
         a = val[k-1];
         break;
      }
   }
   return a;
}


// SparseMatrix: set entry (i,j) to a, 1<=i<=m, 1<=j<=n

void SparseMatrix::setEntry(int i, int j, double a) 
{
   for (int k=rowoff[i-1]; k<rowoff[i]; ++k) {
      if (j==colind[k-1]) {
         val[k-1] = a;
         break;
      }
   }
   return;
}


// SparseMatrix: add a to entry (i,j), 1<=i<=m, 1<=j<=n

void SparseMatrix::addEntry(int i, int j, double a) 
{
   for (int k=rowoff[i-1]; k<rowoff[i]; ++k) {
      if (j==colind[k-1]) {
         val[k-1] += a;
         break;
      }
   }
   return;
}


// SparseMatrix: zero out row 1<=i<=m

void SparseMatrix::zeroutRow(int i) 
{
   for (int k=rowoff[i-1]; k<rowoff[i]; ++k)  val[k-1] = 0.0;
   return;
}


// SparseMatrix: save to a data file 

void SparseMatrix::save2file(char fn[20]) const 
{
   FILE *fp;
   int i, j, k;

   if (NULL==(fp=fopen(fn, "w"))) {
      puts("Open data file failed.  Exit.");
      exit(-1);
   }

   fprintf(fp, "%8d  %8d  %12d\n", m, n, nnz);

   for (i=1; i<=m; ++i) {
      for (k=rowoff[i-1]; k<rowoff[i]; ++k) {
         j = colind[k-1];
         fprintf(fp, "%8d  %8d  %12d  %+18.12f\n", i, j, k, val[k-1]);
      }
   }

   fclose(fp);
   return;
}


// + overloaded: addition of two sparse-matrices of the same sparsity
// Compressed Row Storage format for the sparse matrices

SparseMatrix operator+(const SparseMatrix &A, const SparseMatrix &B) 
{
   SparseMatrix C(A);
   for (int k=0; k<A.nnz; ++k)  C.val[k] = A.val[k] + B.val[k];
   return C;
}


// * overloaded: scalar-sparse-matrix multiplication 
// Compressed Row Storage format for the sparse matrices

SparseMatrix operator*(double a, const SparseMatrix &A) 
{
   SparseMatrix B(A);
   for (int k=0; k<A.nnz; ++k)  B.val[k] = a*A.val[k];
   return B;
}


// * overloaded: sparse-matrix-vector multiplication
// Compressed Row Storage format for the sparse matrix

Vector operator*(const SparseMatrix &A, const Vector &v) 
{
   Vector u(A.m);

   for (int i=0; i<A.m; ++i) 
      for (int k=A.rowoff[i]; k<A.rowoff[i+1]; ++k) 
         u[i] += A.val[k-1]*v(A.colind[k-1]);

   return u;
}


// Get the diagonal (matrix) of a sparse matrix

DiagMatrix diagonal(const SparseMatrix&A) 
{
   if (A.m!=A.n) {
      cout << "Not a square matrix!\n";
      exit(-1);
   }

   DiagMatrix D(A.n);

   for (int i=1; i<=A.n; ++i) {
      for (int k=A.rowoff[i-1]; k<A.rowoff[i]; ++k) {
         if (i==A.colind[k-1]) {
            D.setEntry(i, A.val[k-1]);
            break;
         }
      }
   }

   return D;
}


// SparseBlockMatrix: default constructor

SparseBlockMatrix::SparseBlockMatrix() 
{
   nrb = 0;  ncb = 0;  nnzb = 0;
   drb = 0;  dcb = 0;
   brb = 0;  bcb = 0;
   bro = 0;  bci = 0;  bv = 0;
   m = 0;  n = 0;  nnz = 0;
// rowoff = 0;  colind = 0;  val = 0;
}


// SparseBlockMatrix: another constructor
/*
SparseBlockMatrix::SparseBlockMatrix(int numRowBlks, int numColBlks, 
   int numNzBlks)
{
   nrb = numRowBlks;
   ncb = numColBlks;
   nnzb = numNzBlks;

   bro = new int[nrb+1];
   bci = new int[nnzb];
   bv = new FullMatrix[nnzb];

   drb = 0;  dcb = 0;
   brb = 0;  bcb = 0;
   m = 0;  n = 0;  nnz = 0;
// rowoff = 0;  colind = 0;  val = 0;
}
*/


// SparseBlockMatrix: yet another constructor

SparseBlockMatrix::SparseBlockMatrix(int numRowBlks, int numColBlks, 
   int numNzBlks, int *dimRowBlk, int *dimColBlk, 
   int *blkRowOff, int *blkColInd)
{
   int i, j, k, s, t;

   nrb = numRowBlks;
   ncb = numColBlks;
   nnzb = numNzBlks;

   drb = new int[nrb];
   dcb = new int[ncb]; 
   
   bro = new int[nrb+1];
   bci = new int[nnzb];

   brb = new int[nrb+1];
   bcb = new int[ncb+1];

   bv = new FullMatrix[nnzb];

   brb[0] = 1;
   bcb[0] = 1; 

   m = 0;  n = 0;  nnz = 0;
		
   for (i=0; i<nrb; ++i) {
      drb[i] = dimRowBlk[i];
      bro[i] = blkRowOff[i]; 
      brb[i+1] = brb[i]+drb[i]; 
      m += drb[i];
   }	  

   bro[nrb] = nnzb+1;

   for (j=0; j<ncb; ++j) { 
      dcb[j] = dimColBlk[j];
      bcb[j+1] = bcb[j]+dcb[j];
      n += dcb[j]; 
   }

   for (k=0; k<nnzb; ++k)  bci[k] = blkColInd[k];

   for (s=0; s<nrb; ++s) {
      for (t=bro[s]-1; t<bro[s+1]-1; ++t) {
         bv[t].reshape(drb[s],dcb[bci[t]-1]);
         nnz += drb[s]*(dcb[bci[t]-1]);
      }
   }
}


// SparseBlockMatrix: destructor

SparseBlockMatrix::~SparseBlockMatrix()
{
   nrb = 0;  ncb = 0;  nnzb = 0;
   delete[] drb, dcb;
   delete[] brb, bcb;
   delete[] bro, bci;
   delete[] bv;
   m = 0;  n = 0;  nnz = 0;
// delete[] rowoff, colind, val;
}


// SparseBlockMatrix: reset an (empty) sparse block matrix 

void SparseBlockMatrix::reset(int numRowBlks, int numColBlks, int numNzBlks, 
   int *dimRowBlk, int *dimColBlk, int *blkRowOff, int *blkColInd)
{
   int i, j, k;

   delete[] drb, dcb, brb, bcb, bro, bci, bv;
   nrb = 0;  ncb = 0;  nnzb = 0;
   m = 0;  n = 0;  nnz = 0;

   nrb = numRowBlks;
   ncb = numColBlks;
   nnzb = numNzBlks;

   drb = new int[nrb];
   dcb = new int[ncb];

   for (i=0; i<nrb; ++i)  drb[i] = dimRowBlk[i];
   for (j=0; j<ncb; ++j)  dcb[j] = dimColBlk[j];

   brb = new int[nrb];
   bcb = new int[ncb];

   brb[0] = 1;
   bcb[0] = 1;

   for (i=1; i<nrb; ++i)  brb[i] = brb[i-1] + drb[i-1];
   for (j=1; j<ncb; ++j)  bcb[j] = bcb[j-1] + dcb[j-1];

   bro = new int[nrb+1];
   bci = new int[nnzb];
   bv = new FullMatrix[nnzb];

   for (i=0; i<nrb; ++i)  bro[i] = blkRowOff[i];
   for (k=0; k<nnzb; ++k)  bci[k] = blkColInd[k];

   bro[nrb] = nnzb + 1;

   m = 0;
   n = 0;

   for (i=0; i<nrb; ++i)  m += drb[i];
   for (j=0; j<ncb; ++j)  n += dcb[j];

   nnz = 0;

   for (i=0; i<nrb; ++i) {
      for (k=bro[i]-1; k<bro[i+1]-1; ++k) {
         if (bci[k]!=0) {
            nnz += drb[i]*dcb[bci[k]-1];
            bv[k].reshape(drb[i], dcb[bci[k]-1]);
         }
      }
   }

   return;
}


// SparseBlockMatrix: add A to block (ib,jb)

void SparseBlockMatrix::add(int ib, int jb, const FullMatrix &A)
{
   int bp;  // block position

   for (int k=bro[ib-1]; k<bro[ib]; ++k){
      if (bci[k-1] == jb) {
         bp = k-1;
         break;
      } 
   }

   if (bv[bp].rowSize() != A.rowSize() || bv[bp].colSize() != A.colSize())  
      bv[bp].reshape(A.rowSize(), A.colSize());

   bv[bp] = bv[bp] + A;
}


// SparseBlockMatrix: replace block (ib,jb) by A

void SparseBlockMatrix::replace(int ib, int jb, const FullMatrix &A)
{
   int bp;  // block position

   for (int k=bro[ib-1]; k<bro[ib]; ++k) {
      if (bci[k-1] == jb) {
         bp = k-1;
         break;
      } 
   }

   if (bv[bp].rowSize() != A.rowSize() || bv[bp].colSize() != A.colSize())  
      bv[bp].reshape(A.rowSize(), A.colSize());

   bv[bp] = A;
}


// SparseBlockMatrix: get block (ib,jb)
// Finished by RC 

FullMatrix SparseBlockMatrix::get(int ib, int jb) const 
{
   int bp;  // block position

   for (int k=bro[ib-1]; k<bro[ib]; ++k) {
      if (bci[k-1] == jb) {
         bp = k-1;
         break;
      } 
   }

   return bv[bp];
}


// SparseBlockMatrix: cout for a small size SparseBlockMatrix

std::ostream &operator<<(std::ostream &strm, const SparseBlockMatrix &A)
{
   int i;
   cout << "nrb: " << A.nrb << "\n";
   cout << "ncb: " << A.ncb << "\n";
   cout << "nnzb: " << A.nnzb << "\n";

   cout << "drb: [ ";
   for (i=0; i<A.nrb; ++i)  strm << A.drb[i] << " ";
   cout << "]" <<"\n";

   cout << "dcb: [ ";
   for (i=0; i<A.ncb; ++i)  strm << A.dcb[i] << " ";
   cout << "]" << "\n";

   cout << "bro: [ ";
   for (i=0; i<=A.nrb; ++i)  strm << A.bro[i] << " ";
   cout << "]" << "\n";

   cout << "bci: [ ";
   for (i=0; i<A.nnzb; ++i)  strm << A.bci[i] << " ";
   cout << "]" << "\n";

   cout << "brb: [ ";
   for (i=0; i<=A.nrb; ++i)  strm << A.brb[i] << " ";
   cout << "]" << "\n";

   cout << "bcb: [ ";
   for (i=0; i<=A.ncb; ++i)  strm << A.bcb[i] << " ";
   cout << "]" << "\n";

   cout << "bv: " << "\n";
   for (i=0; i<A.nnzb; ++i)  strm << A.bv[i];
   cout << "\n";

   cout << "m: "  << A.m << "\n";
   cout << "n: "  << A.n << "\n";
   cout << "nnz: "  << A.nnz << "\n";

   return strm;
}


// * overloaded: sparse-block-matrix-vector multiplication

Vector operator*(const SparseBlockMatrix&A, const Vector&v) 
{
   int i, j, k, l, uPos, vPos;

   Vector u(A.m);
   for (i=0; i<A.m; ++i)  u[i]=0.0;

   for (i=0; i<A.nrb; ++i){
      for (j=A.bro[i]-1; j<A.bro[i+1]-1; ++j){
         uPos = A.brb[i]-1;
         for (k=1; k<=A.drb[i]; ++k){
            vPos = A.bcb[A.bci[j]-1]-1;
            for (l=1; l<=A.dcb[A.bci[j]-1]; ++l){ 
               u[uPos] += A.bv[j](k,l)*v[vPos];
               ++vPos;
            }
            ++uPos;
         }
      }
   }

   return u;
}


// BlockDiagMatrix: 
// a constructor based on a block sparse matrix

BlockDiagMatrix::BlockDiagMatrix(const SparseBlockMatrix &A) 
{
   int dc, dr;  // dimension row, dimension column

   nb = A.numRowBlks();
   bb = new int[nb+1];
   db = new int[nb+1];
   bv = new FullMatrix[nb+1];
   n = 0;

   bb[0]=0;
   db[0]=0;

   for (int ib=1; ib<=nb; ++ib) {
      dr = A.get(ib,ib).rowSize();
      dc = A.get(ib,ib).colSize();

      if (dr != dc){
         cout << "Error\n";
         exit(EXIT_FAILURE);          
      }

      bv[ib].reshape(dr, dc);
      bv[ib] = A.get(ib,ib);        

      db[ib] = dr;       
      bb[ib] = n;

      n += dr;
   }
}
