// fve2.h
//
// Finite volume elements in 2-dim
//
// Jiangguo (James) Liu, ColoState, 01/2008--10/2011

#ifndef FVE2_H
#define FVE2_H

#include "matrix.h"
#include "PtVec2d.h"

class Line2d {
public:
  Line2d() {;}
  Line2d(PtVec2d A, PtVec2d B) {P1=A; P2=B;}  // Constructor
  Line2d(const Line2d &L) {P1=L.P1; P2=L.P2;}  // Copy constructor
  Line2d &operator=(const Line2d&);  // Copy assignment
  double length() const {return dist(P1,P2);}
  PtVec2d midpoint() const {return 0.5*(P1+P2);}
  PtVec2d normal() const;
  // Mapping from [0,1] with natural coordinate a
  PtVec2d mapping(double a) const {return (1-a)*P1+a*P2;} 
  // Mapping from [-1,1] with natural coordinate a
  PtVec2d mapping1(double a) const {return (1-a)/2*P1+(1+a)/2*P2;}   
private: 
  PtVec2d P1, P2;
};

class Quadri2 {
public: 
  Quadri2() {;}
  Quadri2(PtVec2d A, PtVec2d B, PtVec2d C, PtVec2d D);
  Quadri2(const Quadri2 &Q);
  Quadri2 &operator=(const Quadri2&Q);
  PtVec2d mapping(double X, double Y) const;
  double jacobian(double X, double Y) const;
  PtVec2d invMapping(PtVec2d P) const;
  double locBasisFxn(double X, double Y, int label) const;
  PtVec2d locBasisFxnGrad(double X, double Y, int label) const;
private: 
  PtVec2d P1, P2, P3, P4;
  double x1, x2, x3, x4;
  double y1, y2, y3, y4;
  double a[4], b[4];
};

class DualQuadri {
public:
  DualQuadri() {;}
  DualQuadri(PtVec2d A, PtVec2d B, PtVec2d C, PtVec2d D);
  DualQuadri(const DualQuadri &Q);
  DualQuadri &operator=(const DualQuadri&Q);
  PtVec2d center() const {return 0.25*(P1+P2+P3+P4);} 
  double area() const;
  PtVec2d mapping1(double X, double Y) const;
  double jacobian1(double X, double Y) const;
private:
  PtVec2d P1, P2, P3, P4;
  double x1, x2, x3, x4;
  double y1, y2, y3, y4;
  double a1[4], b1[4];
};

#endif  // FVE2_H

// Quadri2: 
// Quadrilateral biquadratic elements as primal volumes
// Copy constructor 
// Copy assignment
// The bilinear mapping from the unit square with coordinates X,Y
// The Jacobian of the above bilinear mapping with coordinates X,Y
// The inverse mapping back to the unit square [0,1]^2 

// DualQuadri: 
// Dual quadrilateral elements as control volumes  
// The bilinear mapping from [-1,1]^2 with coordinates X,Y
// The Jacobian of the above bilinear mapping1 with coordinates X,Y
