// fve2.cpp
//
// Finite volume elements in 2-dim
//
// Jiangguo (James) Liu, ColoState, 01/2008--10/2011

#include <cmath>
#include "fve2.h"
#include "matrix.h"
#include "PtVec2d.h"


// Line2d: copy assignment 

Line2d &Line2d::operator=(const Line2d &L) 
{
  if (this != &L) {
    P1 = L.P1;
    P2 = L.P2;
  }
  return *this;
}


// Line2d: a normal vector

PtVec2d Line2d::normal() const 
{
  double x = P2.xCrd() - P1.xCrd();
  double y = P2.yCrd() - P1.yCrd();
  double tmp = sqrt(x*x+y*y);
  return PtVec2d(y/tmp, -x/tmp);
}


// Quadri2: a constructor

Quadri2::Quadri2(PtVec2d A, PtVec2d B, PtVec2d C, PtVec2d D) 
{
  P1 = A;  P2 = B;  P3 = C;  P4 = D;

  x1 = P1.xCrd();  y1 = P1.yCrd();
  x2 = P2.xCrd();  y2 = P2.yCrd();
  x3 = P3.xCrd();  y3 = P3.yCrd();
  x4 = P4.xCrd();  y4 = P4.yCrd();

  a[0] = x1;  a[1] = x2 - x1;  a[2] = x4 - x1;  a[3] = x1 - x2 + x3 - x4;
  b[0] = y1;  b[1] = y2 - y1;  b[2] = y4 - y1;  b[3] = y1 - y2 + y3 - y4;
}


// Quadri2: copy constructor 

Quadri2::Quadri2(const Quadri2 &Q) 
{
  P1 = Q.P1;  P2 = Q.P2;  P3 = Q.P3;  P4 = Q.P4;

  x1 = P1.xCrd();  y1 = P1.yCrd();
  x2 = P2.xCrd();  y2 = P2.yCrd();
  x3 = P3.xCrd();  y3 = P3.yCrd();
  x4 = P4.xCrd();  y4 = P4.yCrd();

  a[0] = x1;  a[1] = x2 - x1;  a[2] = x4 - x1;  a[3] = x1 - x2 + x3 - x4;
  b[0] = y1;  b[1] = y2 - y1;  b[2] = y4 - y1;  b[3] = y1 - y2 + y3 - y4;
};


// Quadri2: copy assignment 

Quadri2 &Quadri2::operator=(const Quadri2 &Q) 
{
  if (this != &Q) {
    P1 = Q.P1;  P2 = Q.P2;  P3 = Q.P3;  P4 = Q.P4;
  }

  x1 = P1.xCrd();  y1 = P1.yCrd();
  x2 = P2.xCrd();  y2 = P2.yCrd();
  x3 = P3.xCrd();  y3 = P3.yCrd();
  x4 = P4.xCrd();  y4 = P4.yCrd();

  a[0] = x1;  a[1] = x2 - x1;  a[2] = x4 - x1;  a[3] = x1 - x2 + x3 - x4;
  b[0] = y1;  b[1] = y2 - y1;  b[2] = y4 - y1;  b[3] = y1 - y2 + y3 - y4;

  return *this;
}


// Quadri2: Mapping from the unit square [0,1]^2 with coordinates X,Y

PtVec2d Quadri2::mapping(double X, double Y) const 
{
  double x = a[0] + a[1]*X + a[2]*Y + a[3]*X*Y;
  double y = b[0] + b[1]*X + b[2]*Y + b[3]*X*Y;
  return PtVec2d(x, y);
}


// Quadri2: the Jacobian of the bilinear mapping from the unit square [0,1]^2
// Notice that (X,Y) are the coordinates in the unit square [0,1]^2 

double Quadri2::jacobian(double X, double Y) const 
{
  double jac = a[1]*b[2]-a[2]*b[1];
  jac += (a[1]*b[3]-a[3]*b[1])*X + (a[3]*b[2]-a[2]*b[3])*Y;
  return jac;
}


// Quadri2: the 9 local basis functions
// Notice that (X,Y) are the coordinates in the unit square [0,1]^2 

double Quadri2::locBasisFxn(double X, double Y, int label) const 
{
  double Z = 0;

  switch (label) { 
  case 1:   Z = (X-1)*(2*X-1)*(Y-1)*(2*Y-1);  break;
  case 2:   Z =     X*(2*X-1)*(Y-1)*(2*Y-1);  break;
  case 3:   Z =     X*(2*X-1)*    Y*(2*Y-1);  break;
  case 4:   Z = (X-1)*(2*X-1)*    Y*(2*Y-1);  break;
  case 5:   Z =   4*X*(1-X)  *(Y-1)*(2*Y-1);  break;
  case 6:   Z =     X*(2*X-1)*  4*Y*(1-Y);    break;
  case 7:   Z =   4*X*(1-X)  *    Y*(2*Y-1);  break;
  case 8:   Z = (X-1)*(2*X-1)*  4*Y*(1-Y);    break;
  case 9:   Z =   4*X*(1-X)  *  4*Y*(1-Y);    break;
  default:  Z = 0;                            break;
  }

  return Z;
}


// Quadri2: the 9 local basis functions: gradients
// Notice that (X,Y) are the coordinates in the unit square [0,1]^2 

PtVec2d Quadri2::locBasisFxnGrad(double X, double Y, int label) const 
{
  double UX=0, UY=0;

  switch (label) { 
  case 1:  UX=(4*X-3)*(Y-1)*(2*Y-1);  UY=(X-1)*(2*X-1)*(4*Y-3);  break;
  case 2:  UX=(4*X-1)*(Y-1)*(2*Y-1);  UY=    X*(2*X-1)*(4*Y-3);  break;
  case 3:  UX=(4*X-1)*    Y*(2*Y-1);  UY=    X*(2*X-1)*(4*Y-1);  break;
  case 4:  UX=(4*X-3)*    Y*(2*Y-1);  UY=(X-1)*(2*X-1)*(4*Y-1);  break;
  case 5:  UX=(4-8*X)*(Y-1)*(2*Y-1);  UY=    4*X*(1-X)*(4*Y-3);  break;
  case 6:  UX=(4*X-1)*  4*Y*(1-Y);    UY=    X*(2*X-1)*(4-8*Y);  break;
  case 7:  UX=(4-8*X)*    Y*(2*Y-1);  UY=    4*X*(1-X)*(4*Y-1);  break;
  case 8:  UX=(4*X-3)*  4*Y*(1-Y);    UY=(X-1)*(2*X-1)*(4-8*Y);  break;
  case 9:  UX=(4-8*X)*  4*Y*(1-Y);    UY=    4*X*(1-X)*(4-8*Y);  break;
  default: UX=0;                      UY=0;                      break;
  }

  double jac = jacobian(X,Y);

  double ux = (1/jac)* ( (b[2]+b[3]*X)*UX - (b[1]+b[3]*Y)*UY);
  double uy = (1/jac)* (-(a[2]+a[3]*X)*UX + (a[1]+a[3]*Y)*UY);

  return PtVec2d(ux,uy);
}


// The inverse mapping to the unit square [0,1]^2 
// based on the Newton's method for nonlinear algebraic systems 

PtVec2d Quadri2::invMapping(PtVec2d P) const 
{
  double x = P.xCrd();
  double y = P.yCrd();
  double Xn, Yn, Xn1, Yn1;
  double alpha, beta, gamma, delta, det, f1n, f2n;
  double epsilon = 1E-12;  // Acceptable accuracy?

  Xn = 0.5;
  Yn = 0.5;

  while (1) {
    f1n = a[3]*Xn*Yn + a[2]*Yn + a[1]*Xn + a[0] - x;
    f2n = b[3]*Xn*Yn + b[2]*Yn + b[1]*Xn + b[0] - y;

    alpha = a[3]*Yn + a[1];
    beta  = a[3]*Xn + a[2];
    gamma = b[3]*Yn + b[1];
    delta = b[3]*Xn + b[2];
    det = alpha*delta - beta*gamma;

    Xn1 = Xn - (1/det)*( delta*f1n - beta*f2n);
    Yn1 = Yn - (1/det)*(-gamma*f1n + alpha*f2n);

    if (fabs(Xn1-Xn)<epsilon && fabs(Yn1-Yn)<epsilon)  break;

    Xn = Xn1;
    Yn = Yn1;
  }

  if (Xn1<0)  Xn1 = 0;
  if (Xn1>1)  Xn1 = 1;
  if (Yn1<0)  Yn1 = 0;
  if (Yn1>1)  Yn1 = 1;

  return PtVec2d(Xn1,Yn1);
}


// DualQuadri: a constructor

DualQuadri::DualQuadri(PtVec2d A, PtVec2d B, PtVec2d C, PtVec2d D) 
{
  P1 = A;  P2 = B;  P3 = C;  P4 = D;

  x1 = P1.xCrd();  y1 = P1.yCrd();
  x2 = P2.xCrd();  y2 = P2.yCrd();
  x3 = P3.xCrd();  y3 = P3.yCrd();
  x4 = P4.xCrd();  y4 = P4.yCrd();

  a1[0] = 0.25*( x1 + x2 + x3 + x4);
  a1[1] = 0.25*(-x1 + x2 + x3 - x4);
  a1[2] = 0.25*(-x1 - x2 + x3 + x4);
  a1[3] = 0.25*( x1 - x2 + x3 - x4);

  b1[0] = 0.25*( y1 + y2 + y3 + y4);
  b1[1] = 0.25*(-y1 + y2 + y3 - y4);
  b1[2] = 0.25*(-y1 - y2 + y3 + y4);
  b1[3] = 0.25*( y1 - y2 + y3 - y4);
}


// DualQuadri: copy constructor 

DualQuadri::DualQuadri(const DualQuadri &Q) 
{
  P1 = Q.P1;  P2 = Q.P2;  P3 = Q.P3;  P4 = Q.P4;

  x1 = P1.xCrd();  y1 = P1.yCrd();
  x2 = P2.xCrd();  y2 = P2.yCrd();
  x3 = P3.xCrd();  y3 = P3.yCrd();
  x4 = P4.xCrd();  y4 = P4.yCrd();

  a1[0] = 0.25*( x1 + x2 + x3 + x4);
  a1[1] = 0.25*(-x1 + x2 + x3 - x4);
  a1[2] = 0.25*(-x1 - x2 + x3 + x4);
  a1[3] = 0.25*( x1 - x2 + x3 - x4);

  b1[0] = 0.25*( y1 + y2 + y3 + y4);
  b1[1] = 0.25*(-y1 + y2 + y3 - y4);
  b1[2] = 0.25*(-y1 - y2 + y3 + y4);
  b1[3] = 0.25*( y1 - y2 + y3 - y4);
};


// Quadri: copy assignment 

DualQuadri &DualQuadri::operator=(const DualQuadri &Q) 
{
  if (this != &Q) {
    P1 = Q.P1;  P2 = Q.P2;  P3 = Q.P3;  P4 = Q.P4;
  }

  x1 = P1.xCrd();  y1 = P1.yCrd();
  x2 = P2.xCrd();  y2 = P2.yCrd();
  x3 = P3.xCrd();  y3 = P3.yCrd();
  x4 = P4.xCrd();  y4 = P4.yCrd();

  a1[0] = 0.25*( x1 + x2 + x3 + x4);
  a1[1] = 0.25*(-x1 + x2 + x3 - x4);
  a1[2] = 0.25*(-x1 - x2 + x3 + x4);
  a1[3] = 0.25*( x1 - x2 + x3 - x4);

  b1[0] = 0.25*( y1 + y2 + y3 + y4);
  b1[1] = 0.25*(-y1 + y2 + y3 - y4);
  b1[2] = 0.25*(-y1 - y2 + y3 + y4);
  b1[3] = 0.25*( y1 - y2 + y3 - y4);

  return *this;
}


// DualQuadri: the area

double DualQuadri::area() const 
{
  return 0.5*fabs((x2-x1)*(y3-y1)-(x3-x1)*(y2-y1)) 
    + 0.5*fabs((x3-x1)*(y4-y1)-(x4-x1)*(y3-y1));
}


// DualQuadri: the bilinear mapping from [-1,1]^2 with coordinates X,Y

PtVec2d DualQuadri::mapping1(double X, double Y) const 
{
  double x = a1[0] + a1[1]*X + a1[2]*Y + a1[3]*X*Y;
  double y = b1[0] + b1[1]*X + b1[2]*Y + a1[3]*X*Y;
  return PtVec2d(x,y);
}


// DualQuadri: the Jacobian 
// for the bilinear mapping from [-1,1]^2 with coordinates X,Y

double DualQuadri::jacobian1(double X, double Y) const 
{
  double z = (a1[1]*b1[2]-a1[2]*b1[1]);
  z += (a1[1]*b1[3]-a1[3]*b1[1])*X + (a1[3]*b1[2]-a1[2]*b1[3])*Y;
  return z;
}


// Standard 2-dim local quadratic basis functions on the unit square 
// label 1,2,3,4 -- Vertex P1=(0,0), P2=(1,0), P3=(1,1), P4=(0,1)
// label 5,6 -- Edge midpoint M1=(0.5,0), M2=(1,0.5)
// label 7,8 -- Edge midpoint M3=(0.5 1), M4=(0,0.5)
// label 9 -- Center (0.5,0.5)
/*
double std2dLocBasisFxnQua(PtVec2d P, int label)
{
  double x = P.xCrd();
  double y = P.yCrd();
  double z = 0;

  switch (label) { 
  case 1:   z = (x-1)*(2*x-1)*(y-1)*(2*y-1);  break;
  case 2:   z =     x*(2*x-1)*(y-1)*(2*y-1);  break;
  case 3:   z =     x*(2*x-1)*    y*(2*y-1);  break;
  case 4:   z = (x-1)*(2*x-1)*    y*(2*y-1);  break;
  case 5:   z =   4*x*(1-x)  *(y-1)*(2*y-1);  break;
  case 6:   z =     x*(2*x-1)*  4*y*(1-y);    break;
  case 7:   z =   4*x*(1-x)  *    y*(2*y-1);  break;
  case 8:   z = (x-1)*(2*x-1)*  4*y*(1-y);    break;
  case 9:   z =   4*x*(1-x)  *  4*y*(1-y);    break;
  default:  z = 0;                            break;
  }

  return z;
}
*/


// standard 2-dim local quadratic basis functions on the unit square 
// the gradients
/*
PtVec2d std2dLocBasisFxnQuaGrad(PtVec2d P, int label) 
{
  double x = P.xCrd();
  double y = P.yCrd();
  double u = 0;
  double v = 0;

  switch (label) { 
  case 1:   u=(4*x-3)*(y-1)*(2*y-1);  v=(x-1)*(2*x-1)*(4*y-3);  break;
  case 2:   u=(4*x-1)*(y-1)*(2*y-1);  v=    x*(2*x-1)*(4*y-3);  break;
  case 3:   u=(4*x-1)*    y*(2*y-1);  v=    x*(2*x-1)*(4*y-1);  break;
  case 4:   u=(4*x-3)*    y*(2*y-1);  v=(x-1)*(2*x-1)*(4*y-1);  break;
  case 5:   u=(4-8*x)*(y-1)*(2*y-1);  v=    4*x*(1-x)*(4*y-3);  break;
  case 6:   u=(4*x-1)*  4*y*(1-y);    v=    x*(2*x-1)*(4-8*y);  break;
  case 7:   u=(4-8*x)*    y*(2*y-1);  v=    4*x*(1-x)*(4*y-1);  break;
  case 8:   u=(4*x-3)*  4*y*(1-y);    v=(x-1)*(2*x-1)*(4-8*y);  break;
  case 9:   u=(4-8*x)*  4*y*(1-y);    v=    4*x*(1-x)*(4-8*y);  break;
  default:  u=0;                      v=0;                      break;
  }

  return PtVec2d(u,v);
}
*/
