// Quadri2Mesh.cpp
//
// (2-dim) quadrilateral meshes 
// affine biquadratic finite volme approximation 
//
// Jiangguo (James) Liu, ColoState, 01/2008--10/2011

#include <cmath>
#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include "fve2.h"
#include "GaussQuad.h"
#include "matrix.h"
#include "PtVec2d.h"
#include "Quadri2Mesh.h"
#include "vector.h"
using namespace std;


// Quadri2Mesh: a constructor 
// from a data file with a specific structure

Quadri2Mesh::Quadri2Mesh(char *filename) 
{
  FILE *fp;
  int i, j, k, lblVrtx[10];
  double x, y;
  PtVec2d P[10];

  if (NULL==(fp=fopen(filename, "r"))) {
    puts("Open data file failed.  Exit.");
    exit(-1);
  }

  fscanf(fp, "%7d", &nx);
  fscanf(fp, "%7d", &ny);

  nd = new PtVec2d[(2*nx+1)*(2*ny+1)];

// Only the 1st type node info is available in the file 

  for (j=0; j<=ny; ++j) {
    for (i=0; i<=nx; ++i) {
      fscanf(fp, "%lf  %lf", &x, &y);
      nd[j*(nx+1)+i] = PtVec2d(x, y);
    }
  }

  fclose(fp);

  n1 = (nx+1)*(ny+1);  // Number of original nodes
  n2 = nx*(ny+1);      // Number of horizontal edge midpoints
  n3 = (nx+1)*ny;      // Number of vertical edge midpoints
  n4 = nx*ny;          // Number of quadrilateral centers

  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      lblVrtx[0] = 0;  // But not used 
      lblVrtx[1] = (j-1)*(nx+1) + (i-1) + 1;
      lblVrtx[2] = lblVrtx[1] + 1;
      lblVrtx[3] = lblVrtx[2] + (nx+1);
      lblVrtx[4] = lblVrtx[3] - 1;
      lblVrtx[5] = n1 + (j-1)*nx + i;
      lblVrtx[7] = lblVrtx[5] + nx;
      lblVrtx[8] = n1 + n2 + (j-1)*(nx+1) + (i-1) + 1;
      lblVrtx[6] = lblVrtx[8] + 1;
      lblVrtx[9] = n1 + n2 + n3 + (j-1)*nx + i;

      for (k=1; k<=4; ++k)  P[k] = nd[lblVrtx[k]-1];

      P[5] = 0.5*(P[1]+P[2]);
      P[6] = 0.5*(P[2]+P[3]);
      P[7] = 0.5*(P[3]+P[4]);
      P[8] = 0.5*(P[4]+P[1]);
      P[9] = 0.25*(P[1]+P[2]+P[3]+P[4]);

      for (k=5; k<=9; ++k)  nd[lblVrtx[k]-1] = P[k];
    }
  }
}


// Quadri2Mesh: getting the label of a node

int Quadri2Mesh::getLblNd(int i, int j, int type) const 
{
  int label;  // Starting at 1

  switch (type) {
  case 1:  // 0<=i<=nx, 0<=j<=ny
    label = j*(nx+1) + i + 1;
    break;
  case 2:  // 1<=i<=nx, 0<=j<=ny
    label = n1 + j*nx + i;
    break;
  case 3:  // 0<=i<=nx, 1<=j<=ny
    label = n1 + n2 + (j-1)*(nx+1) + i + 1;
    break;
  case 4:  // 1<=i<=nx, 1<=j<=ny
    label = n1 + n2 + n3 + (j-1)*nx + i;
    break;
  default:  // If something is wrong
    label = 0;
    break;
  }

  return label;
}


// Quadri2Mesh: getting the label of a dual volume

int Quadri2Mesh::getLblDualVol(int i, int j, int type) const 
{
  int label;  // Starting at 1

  switch (type) {
  case 1:  // 0<=i<=nx, 0<=j<=ny
    label = j*(nx+1) + i + 1;
    break;
  case 2:  // 1<=i<=nx, 0<=j<=ny
    label = n1 + j*nx + i;
    break;
  case 3:  // 0<=i<=nx, 1<=j<=ny
    label = n1 + n2 + (j-1)*(nx+1) + i + 1;
    break;
  case 4:  // 1<=i<=nx, 1<=j<=ny
    label = n1 + n2 + n3 + (j-1)*nx + i;
    break;
  default:  // If something is wrong
    label = 0;
    break;
  }

  return label;
}


// Quadri2Mesh: computing an element mass matrix 

FullMatrix Quadri2Mesh::cmptEltMassMat(int i, int j) const 
{
  int ii, jj, k;
  double a = 0.211324865405187;  // 1/(3+sqrt(3))
  double intgrl, jac, z;
  PtVec2d P1, P2, P3, P4;
  PtVec2d P12, P21, P23, P32, P34, P43, P41, P14, PP1, PP2, PP3, PP4;
  PtVec2d P, PP;
  DualQuadri DQ[10];
  FullMatrix eltMassMat(9,9);

  P1 = nd[(j-1)*(nx+1)+i-1];
  P2 = nd[(j-1)*(nx+1)+i];
  P3 = nd[j*(nx+1)+i];
  P4 = nd[j*(nx+1)+i-1];

  P12 = (1-a)*P1 + a*P2;
  P21 = (1-a)*P2 + a*P1;

  P23 = (1-a)*P2 + a*P3;
  P32 = (1-a)*P3 + a*P2;

  P34 = (1-a)*P3 + a*P4;
  P43 = (1-a)*P4 + a*P3;

  P41 = (1-a)*P4 + a*P1;
  P14 = (1-a)*P1 + a*P4;

  PP1 = (1-a)*(1-a)*P1 +     a*(1-a)*P2 +         a*a*P3 +     (1-a)*a*P4;
  PP2 =     a*(1-a)*P1 + (1-a)*(1-a)*P2 +     (1-a)*a*P3 +         a*a*P4;
  PP3 =         a*a*P1 +     (1-a)*a*P2 + (1-a)*(1-a)*P3 +     a*(1-a)*P4;
  PP4 =     (1-a)*a*P1 +         a*a*P2 +     a*(1-a)*P3 + (1-a)*(1-a)*P4;

  Quadri2 Q2(P1, P2, P3, P4);

  // DQ[0] not used
  DQ[1] = DualQuadri(P1,P12,PP1,P14);
  DQ[2] = DualQuadri(P2,P23,PP2,P21);
  DQ[3] = DualQuadri(P3,P34,PP3,P32);
  DQ[4] = DualQuadri(P4,P41,PP4,P43);
  DQ[5] = DualQuadri(P12,P21,PP2,PP1);
  DQ[6] = DualQuadri(P23,P32,PP3,PP2);
  DQ[7] = DualQuadri(P34,P43,PP4,PP3);
  DQ[8] = DualQuadri(P41,P14,PP1,PP4);
  DQ[9] = DualQuadri(PP1,PP2,PP3,PP4);

  // integration of the jj-th local basis function of Q2 
  // on the ii-th dual quadrilateral DQ[ii]

  for (ii=1; ii<=9; ++ii) {
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQR5npts; ++k) {
        P = DQ[ii].mapping1(GQR5wab[k][1],GQR5wab[k][2]);
        PP = Q2.invMapping(P);
        z = Q2.locBasisFxn(PP.xCrd(), PP.yCrd(), jj);
        jac = DQ[ii].jacobian1(GQR5wab[k][1],GQR5wab[k][2]);
        intgrl += z*fabs(jac)*GQR5wab[k][0];
      }
      eltMassMat(ii,jj) = intgrl*4;
    }
  }

  return eltMassMat;
}


// Quadri2Mesh: computing an element stiffness matrix 

FullMatrix Quadri2Mesh::cmptEltStfMat(int i, int j, 
  double (*perm)(double*)) const
{
  int i1, i2, ii, jj, k;
  double a = 0.211324865405187;  // 1/(3+sqrt(3))
  double intgrl;
  double *txy = new double[3];  // txy[0] not used
  double *diff = new double[GQL6npts];
  double *X = new double[GQL6npts];
  double *Y = new double[GQL6npts];
  PtVec2d P1, P2, P3, P4;
  PtVec2d P12, P21, P23, P32, P34, P43, P41, P14, PP1, PP2, PP3, PP4;
  PtVec2d grad, normal, P, V;
  FullMatrix eltStfMat(9,9);

  P1 = nd[(j-1)*(nx+1)+i-1];
  P2 = nd[(j-1)*(nx+1)+i];
  P3 = nd[j*(nx+1)+i];
  P4 = nd[j*(nx+1)+i-1];

  P12 = (1-a)*P1 + a*P2;
  P21 = (1-a)*P2 + a*P1;

  P23 = (1-a)*P2 + a*P3;
  P32 = (1-a)*P3 + a*P2;

  P34 = (1-a)*P3 + a*P4;
  P43 = (1-a)*P4 + a*P3;

  P41 = (1-a)*P4 + a*P1;
  P14 = (1-a)*P1 + a*P4;

  PP1 = (1-a)*(1-a)*P1 +     a*(1-a)*P2 +         a*a*P3 +     (1-a)*a*P4;
  PP2 =     a*(1-a)*P1 + (1-a)*(1-a)*P2 +     (1-a)*a*P3 +         a*a*P4;
  PP3 =         a*a*P1 +     (1-a)*a*P2 + (1-a)*(1-a)*P3 +     a*(1-a)*P4;
  PP4 =     (1-a)*a*P1 +         a*a*P2 +     a*(1-a)*P3 + (1-a)*(1-a)*P4;

  Line2d L;
  Quadri2 Q2(P1, P2, P3, P4);
  DualQuadri DQ[10];
  // DQ[0] not used
  DQ[1] = DualQuadri(P1,P12,PP1,P14);
  DQ[2] = DualQuadri(P2,P23,PP2,P21);
  DQ[3] = DualQuadri(P3,P34,PP3,P32);
  DQ[4] = DualQuadri(P4,P41,PP4,P43);
  DQ[5] = DualQuadri(P12,P21,PP2,PP1);
  DQ[6] = DualQuadri(P23,P32,PP3,PP2);
  DQ[7] = DualQuadri(P34,P43,PP4,PP3);
  DQ[8] = DualQuadri(P41,P14,PP1,PP4);
  DQ[9] = DualQuadri(PP1,PP2,PP3,PP4);

  // Edge P12-PP1
  L = Line2d(P12, PP1);
  normal = L.normal();
  V = L.midpoint() - DQ[1].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = a;
    Y[k] = (a/2)*GQL6wab[k][1] + (a/2);
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 1;  i2 = 5;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge P21-PP2
  L = Line2d(P21, PP2);
  normal = L.normal();
  V = L.midpoint() - DQ[2].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = 1-a;
    Y[k] = (a/2)*GQL6wab[k][1] + (a/2);
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 2;  i2 = 5;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
       grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
       intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge PP1-PP4
  L = Line2d(PP1, PP4);
  normal = L.normal();
  V = L.midpoint() - DQ[8].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = a;
    Y[k] = ((1-2*a)/2)*GQL6wab[k][1] + 0.5;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 8;  i2 = 9;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge PP2-PP3
  L = Line2d(PP2, PP3);
  normal = L.normal();
  V = L.midpoint() - DQ[6].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = 1-a;
    Y[k] = ((1-2*a)/2)*GQL6wab[k][1] + 0.5;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 6;  i2 = 9;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge PP4-P43
  L = Line2d(PP4, P43);
  normal = L.normal();
  V = L.midpoint() - DQ[4].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = a;
    Y[k] = (a/2)*GQL6wab[k][1] + (2-a)/2;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 4;  i2 = 7;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge PP3-P34
  L = Line2d(PP3, P34);
  normal = L.normal();
  V = L.midpoint() - DQ[3].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = 1-a;
    Y[k] = (a/2)*GQL6wab[k][1] + (2-a)/2;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 3;  i2 = 7;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge P14-PP1
  L = Line2d(P14, PP1);
  normal = L.normal();
  V = L.midpoint() - DQ[1].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = (a/2)*GQL6wab[k][1] + a/2;
    Y[k] = a;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 1;  i2 = 8;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge PP1-PP2
  L = Line2d(PP1, PP2);
  normal = L.normal();
  V = L.midpoint() - DQ[5].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = ((1-2*a)/2)*GQL6wab[k][1] + 0.5;
    Y[k] = a;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 5;  i2 = 9;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge PP2-P23
  L = Line2d(PP2, P23);
  normal = L.normal();
  V = L.midpoint() - DQ[2].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = (a/2)*GQL6wab[k][1] + (2-a)/2;
    Y[k] = a;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 2;  i2 = 6;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge P41-PP4
  L = Line2d(P41, PP4);
  normal = L.normal();
  V = L.midpoint() - DQ[4].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = (a/2)*GQL6wab[k][1] + a/2;
    Y[k] = 1-a;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 4;  i2 = 8;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge PP4-PP3
  L = Line2d(PP4, PP3);
  normal = L.normal();
  V = L.midpoint() - DQ[7].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = ((1-2*a)/2)*GQL6wab[k][1] + 0.5;
    Y[k] = 1-a;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 7;  i2 = 9;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // Edge PP3-P32
  L = Line2d(PP3, P32);
  normal = L.normal();
  V = L.midpoint() - DQ[3].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = (a/2)*GQL6wab[k][1] + (2-a)/2;
    Y[k] = 1-a;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  i1 = 3;  i2 = 6;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(i1, jj) += -intgrl;
    eltStfMat(i2, jj) -= -intgrl;
  }

  // On the bottom boundary
  if (j==1) {
  // Edge P1-P12
  L = Line2d(P1, P12);
  normal = L.normal();
  V = L.midpoint() - DQ[1].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = (a/2)*GQL6wab[k][1] + a/2;
    Y[k] = 0;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  ii = 1;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(ii, jj) += -intgrl;
  }

  // Edge P12-P21
  L = Line2d(P12, P21);
  normal = L.normal();
  V = L.midpoint() - DQ[5].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
     X[k] = ((1-2*a)/2)*GQL6wab[k][1] + 0.5;
     Y[k] = 0;
     P = Q2.mapping(X[k], Y[k]);
     txy[1] = P.xCrd();
     txy[2] = P.yCrd();
     diff[k] = perm(txy);
  }
  ii = 5;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(ii, jj) += -intgrl;
  }

  // Edge P21-P2
  L = Line2d(P21, P2);
  normal = L.normal();
  V = L.midpoint() - DQ[2].center();
  if (dotProd(normal,V)<0)  normal = (-1)*normal;
  for (k=0; k<GQL6npts; ++k) {
    X[k] = (a/2)*GQL6wab[k][1] + (2-a)/2;
    Y[k] = 0;
    P = Q2.mapping(X[k], Y[k]);
    txy[1] = P.xCrd();
    txy[2] = P.yCrd();
    diff[k] = perm(txy);
  }
  ii = 2;
  for (jj=1; jj<=9; ++jj) {
    intgrl = 0;
    for (k=0; k<GQL6npts; ++k) {
      grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
      intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
    }
    intgrl *= L.length();
    eltStfMat(ii, jj) += -intgrl;
  }
 }

  // On the top boundary 
  if (j==ny) {
    // Edge P4-P43
    L = Line2d(P4, P43);
    normal = L.normal();
    V = L.midpoint() - DQ[4].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = (a/2)*GQL6wab[k][1] + a/2;
      Y[k] = 1;
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 4;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
        grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
        intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }

    // Edge P43-P34
    L = Line2d(P43, P34);
    normal = L.normal();
    V = L.midpoint() - DQ[7].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = ((1-2*a)/2)*GQL6wab[k][1] + 0.5;
      Y[k] = 1;
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 7;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
        grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
        intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }

    // Edge P34-P3
    L = Line2d(P34, P3);
    normal = L.normal();
    V = L.midpoint() - DQ[3].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = (a/2)*GQL6wab[k][1] + (2-a)/2;
      Y[k] = 1;
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 3;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
        grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
        intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }
  }

  // On the left boundary 
  if (i==1) {
    // Edge P1-P14
    L = Line2d(P1, P14);
    normal = L.normal();
    V = L.midpoint() - DQ[1].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = 0;
      Y[k] = (a/2)*GQL6wab[k][1] + (a/2);
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 1;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
         grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
         intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }

    // Edge P14-P41
    L = Line2d(P14, P41);
    normal = L.normal();
    V = L.midpoint() - DQ[8].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = 0;
      Y[k] = ((1-2*a)/2)*GQL6wab[k][1] + 0.5;
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 8;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
        grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
        intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }

    // Edge P41-P4
    L = Line2d(P41, P4);
    normal = L.normal();
    V = L.midpoint() - DQ[4].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = 0;
      Y[k] = (a/2)*GQL6wab[k][1] + (2-a)/2;
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 4;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
        grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
        intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }
  }

  // On the right boundary 
  if (i==nx) {
    // Edge P2-P23
    L = Line2d(P2, P23);
    normal = L.normal();
    V = L.midpoint() - DQ[2].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = 1;
      Y[k] = (a/2)*GQL6wab[k][1] + (a/2);
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 2;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
        grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
        intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }

    // Edge P23-P32
    L = Line2d(P23, P32);
    normal = L.normal();
    V = L.midpoint() - DQ[6].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = 1;
      Y[k] = ((1-2*a)/2)*GQL6wab[k][1] + 0.5;
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 6;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
        grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
        intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }

    // Edge P32-P3
    L = Line2d(P32, P3);
    normal = L.normal();
    V = L.midpoint() - DQ[3].center();
    if (dotProd(normal,V)<0)  normal = (-1)*normal;
    for (k=0; k<GQL6npts; ++k) {
      X[k] = 1;
      Y[k] = (a/2)*GQL6wab[k][1] + (2-a)/2;
      P = Q2.mapping(X[k], Y[k]);
      txy[1] = P.xCrd();
      txy[2] = P.yCrd();
      diff[k] = perm(txy);
    }
    ii = 3;
    for (jj=1; jj<=9; ++jj) {
      intgrl = 0;
      for (k=0; k<GQL6npts; ++k) {
        grad = Q2.locBasisFxnGrad(X[k], Y[k], jj);
        intgrl += dotProd(diff[k]*grad,normal)*GQL6wab[k][0];
      }
      intgrl *= L.length();
      eltStfMat(ii, jj) += -intgrl;
    }
  }

  delete[] txy;

  return eltStfMat;
}


// Quadri2Mesh: setting up the global mass and stiffness matrices

void Quadri2Mesh::setGlbMats(SparseMatrix &glbMassMat, 
  SparseMatrix &glbStfMat)
{
  int i, j, lbl, lrow;

  int numRows = getNumDualVols();
  int numCols = getNumNds();
  int *numEntrsInRow = new int[numRows];
  int **colIdx = new int*[numRows];

  // the sparsity info of the global mass/stiffness matrices
  // Dual volumes: Type 1, interior

  for (j=1; j<=(ny-1); ++j) {
    for (i=1; i<=(nx-1); ++i) {
      lbl = getLblDualVol(i, j, 1);
      lrow = lbl - 1;
      numEntrsInRow[lrow] = 25;
      colIdx[lrow] = new int[25];

      colIdx[lrow][0] = getLblNd(i-1, j-1, 1);
      colIdx[lrow][1] = getLblNd(i,   j-1, 1);
      colIdx[lrow][2] = getLblNd(i+1, j-1, 1);
      colIdx[lrow][3] = getLblNd(i-1, j,   1);
      colIdx[lrow][4] = getLblNd(i,   j,   1);
      colIdx[lrow][5] = getLblNd(i+1, j,   1);
      colIdx[lrow][6] = getLblNd(i-1, j+1, 1);
      colIdx[lrow][7] = getLblNd(i,   j+1, 1);
      colIdx[lrow][8] = getLblNd(i+1, j+1, 1);

      colIdx[lrow][9] = getLblNd(i,    j-1, 2);
      colIdx[lrow][10] = getLblNd(i+1, j-1, 2);
      colIdx[lrow][11] = getLblNd(i,   j,   2);
      colIdx[lrow][12] = getLblNd(i+1, j,   2);
      colIdx[lrow][13] = getLblNd(i,   j+1, 2);
      colIdx[lrow][14] = getLblNd(i+1, j+1, 2);

      colIdx[lrow][15] = getLblNd(i-1, j,   3);
      colIdx[lrow][16] = getLblNd(i,   j,   3);
      colIdx[lrow][17] = getLblNd(i+1, j,   3);
      colIdx[lrow][18] = getLblNd(i-1, j+1, 3);
      colIdx[lrow][19] = getLblNd(i,   j+1, 3);
      colIdx[lrow][20] = getLblNd(i+1, j+1, 3);

      colIdx[lrow][21] = getLblNd(i,   j,   4);
      colIdx[lrow][22] = getLblNd(i+1, j,   4);
      colIdx[lrow][23] = getLblNd(i,   j+1, 4);
      colIdx[lrow][24] = getLblNd(i+1, j+1, 4);
    }
  }

  // Dual volumes: Type 1, on the boundaries parallel to the x-axis

  j = 0;
  for (i=1; i<=(nx-1); ++i) {
    lbl = getLblDualVol(i, j, 1);
    lrow = lbl - 1;
    numEntrsInRow[lrow] = 15;
    colIdx[lrow] = new int[15];

    colIdx[lrow][0] = getLblNd(i-1, j,   1);
    colIdx[lrow][1] = getLblNd(i,   j,   1);
    colIdx[lrow][2] = getLblNd(i+1, j,   1);
    colIdx[lrow][3] = getLblNd(i-1, j+1, 1);
    colIdx[lrow][4] = getLblNd(i,   j+1, 1);
    colIdx[lrow][5] = getLblNd(i+1, j+1, 1);

    colIdx[lrow][6] = getLblNd(i,   j,   2);
    colIdx[lrow][7] = getLblNd(i+1, j,   2);
    colIdx[lrow][8] = getLblNd(i,   j+1, 2);
    colIdx[lrow][9] = getLblNd(i+1, j+1, 2);

    colIdx[lrow][10] = getLblNd(i-1, j+1, 3);
    colIdx[lrow][11] = getLblNd(i,   j+1, 3);
    colIdx[lrow][12] = getLblNd(i+1, j+1, 3);

    colIdx[lrow][13] = getLblNd(i,   j+1, 4);
    colIdx[lrow][14] = getLblNd(i+1, j+1, 4);
  }

  j = ny;
  for (i=1; i<=(nx-1); ++i) {
    lbl = getLblDualVol(i, j, 1);
    lrow = lbl - 1;
    numEntrsInRow[lrow] = 15;
    colIdx[lrow] = new int[15];

    colIdx[lrow][0] = getLblNd(i-1, j-1, 1);
    colIdx[lrow][1] = getLblNd(i,   j-1, 1);
    colIdx[lrow][2] = getLblNd(i+1, j-1, 1);
    colIdx[lrow][3] = getLblNd(i-1, j,   1);
    colIdx[lrow][4] = getLblNd(i,   j,   1);
    colIdx[lrow][5] = getLblNd(i+1, j,   1);

    colIdx[lrow][6] = getLblNd(i,   j-1, 2);
    colIdx[lrow][7] = getLblNd(i+1, j-1, 2);
    colIdx[lrow][8] = getLblNd(i,   j,   2);
    colIdx[lrow][9] = getLblNd(i+1, j,   2);

    colIdx[lrow][10] = getLblNd(i-1, j, 3);
    colIdx[lrow][11] = getLblNd(i,   j, 3);
    colIdx[lrow][12] = getLblNd(i+1, j, 3);

    colIdx[lrow][13] = getLblNd(i,   j, 4);
    colIdx[lrow][14] = getLblNd(i+1, j, 4);
  }

  // Dual volumes: Type 1, on the boundaries parallel to the y-axis

  i = 0;
  for (j=1; j<=(ny-1); ++j) {
    lbl = getLblDualVol(i, j, 1);
    lrow = lbl - 1;
    numEntrsInRow[lrow] = 15;
    colIdx[lrow] = new int[15];

    colIdx[lrow][0] = getLblNd(i,   j-1, 1);
    colIdx[lrow][1] = getLblNd(i+1, j-1, 1);
    colIdx[lrow][2] = getLblNd(i,   j,   1);
    colIdx[lrow][3] = getLblNd(i+1, j,   1);
    colIdx[lrow][4] = getLblNd(i,   j+1, 1);
    colIdx[lrow][5] = getLblNd(i+1, j+1, 1);

    colIdx[lrow][6] = getLblNd(i+1, j-1, 2);
    colIdx[lrow][7] = getLblNd(i+1, j,   2);
    colIdx[lrow][8] = getLblNd(i+1, j+1, 2);

    colIdx[lrow][9] =  getLblNd(i,   j,   3);
    colIdx[lrow][10] = getLblNd(i+1, j,   3);
    colIdx[lrow][11] = getLblNd(i,   j+1, 3);
    colIdx[lrow][12] = getLblNd(i+1, j+1, 3);

    colIdx[lrow][13] = getLblNd(i+1, j,   4);
    colIdx[lrow][14] = getLblNd(i+1, j+1, 4);
  }

  i = nx;
  for (j=1; j<=(ny-1); ++j) {
    lbl = getLblDualVol(i, j, 1);
    lrow = lbl - 1;
    numEntrsInRow[lrow] = 15;
    colIdx[lrow] = new int[15];

    colIdx[lrow][0] = getLblNd(i-1, j-1, 1);
    colIdx[lrow][1] = getLblNd(i,   j-1, 1);
    colIdx[lrow][2] = getLblNd(i-1, j,   1);
    colIdx[lrow][3] = getLblNd(i,   j,   1);
    colIdx[lrow][4] = getLblNd(i-1, j+1, 1);
    colIdx[lrow][5] = getLblNd(i,   j+1, 1);

    colIdx[lrow][6] = getLblNd(i, j-1, 2);
    colIdx[lrow][7] = getLblNd(i, j,   2);
    colIdx[lrow][8] = getLblNd(i, j+1, 2);

    colIdx[lrow][9] =  getLblNd(i-1, j,   3);
    colIdx[lrow][10] = getLblNd(i,   j,   3);
    colIdx[lrow][11] = getLblNd(i-1, j+1, 3);
    colIdx[lrow][12] = getLblNd(i,   j+1, 3);

    colIdx[lrow][13] = getLblNd(i, j,   4);
    colIdx[lrow][14] = getLblNd(i, j+1, 4);
  }

  // Dual volumes: Type 1, four corners 

  i = 0;  j = 0;
  lbl = getLblDualVol(i, j, 1);
  lrow = lbl - 1;
  numEntrsInRow[lrow] = 9;
  colIdx[lrow] = new int[9];
  colIdx[lrow][0] = getLblNd(0, 0, 1);
  colIdx[lrow][1] = getLblNd(1, 0, 1);
  colIdx[lrow][2] = getLblNd(0, 1, 1);
  colIdx[lrow][3] = getLblNd(1, 1, 1);
  colIdx[lrow][4] = getLblNd(1, 0, 2);
  colIdx[lrow][5] = getLblNd(1, 1, 2);
  colIdx[lrow][6] = getLblNd(0, 1, 3);
  colIdx[lrow][7] = getLblNd(1, 1, 3);
  colIdx[lrow][8] = getLblNd(1, 1, 4);

  i = nx;  j = 0;
  lbl = getLblDualVol(i, j, 1);
  lrow = lbl - 1;
  numEntrsInRow[lrow] = 9;
  colIdx[lrow] = new int[9];
  colIdx[lrow][0] = getLblNd(nx-1, 0, 1);
  colIdx[lrow][1] = getLblNd(nx,   0, 1);
  colIdx[lrow][2] = getLblNd(nx-1, 1, 1);
  colIdx[lrow][3] = getLblNd(nx,   1, 1);
  colIdx[lrow][4] = getLblNd(nx,   0, 2);
  colIdx[lrow][5] = getLblNd(nx,   1, 2);
  colIdx[lrow][6] = getLblNd(nx-1, 1, 3);
  colIdx[lrow][7] = getLblNd(nx,   1, 3);
  colIdx[lrow][8] = getLblNd(nx,   1, 4);

  i = 0;  j = ny;
  lbl = getLblDualVol(i, j, 1);
  lrow = lbl - 1;
  numEntrsInRow[lrow] = 9;
  colIdx[lrow] = new int[9];
  colIdx[lrow][0] = getLblNd(0, ny-1, 1);
  colIdx[lrow][1] = getLblNd(1, ny-1, 1);
  colIdx[lrow][2] = getLblNd(0, ny,   1);
  colIdx[lrow][3] = getLblNd(1, ny,   1);
  colIdx[lrow][4] = getLblNd(1, ny-1, 2);
  colIdx[lrow][5] = getLblNd(1, ny,   2);
  colIdx[lrow][6] = getLblNd(0, ny,   3);
  colIdx[lrow][7] = getLblNd(1, ny,   3);
  colIdx[lrow][8] = getLblNd(1, ny,   4);

  i = nx;  j = ny;
  lbl = getLblDualVol(i, j, 1);
  lrow = lbl - 1;
  numEntrsInRow[lrow] = 9;
  colIdx[lrow] = new int[9];
  colIdx[lrow][0] = getLblNd(nx-1, ny-1, 1);
  colIdx[lrow][1] = getLblNd(nx,   ny-1, 1);
  colIdx[lrow][2] = getLblNd(nx-1, ny,   1);
  colIdx[lrow][3] = getLblNd(nx,   ny,   1);
  colIdx[lrow][4] = getLblNd(nx,   ny-1, 2);
  colIdx[lrow][5] = getLblNd(nx,   ny,   2);
  colIdx[lrow][6] = getLblNd(nx-1, ny,   3);
  colIdx[lrow][7] = getLblNd(nx,   ny,   3);
  colIdx[lrow][8] = getLblNd(nx,   ny,   4);

  // Dual volumes: Type 2, not on the boundaries 

  for (j=1; j<=(ny-1); ++j) {
    for (i=1; i<=nx; ++i) {
      lbl = getLblDualVol(i, j, 2);
      lrow = lbl - 1;
      numEntrsInRow[lrow] = 15;
      colIdx[lrow] = new int[15];

      colIdx[lrow][0] = getLblNd(i-1, j-1, 1);
      colIdx[lrow][1] = getLblNd(i,   j-1, 1);
      colIdx[lrow][2] = getLblNd(i-1, j,   1);
      colIdx[lrow][3] = getLblNd(i,   j,   1);
      colIdx[lrow][4] = getLblNd(i-1, j+1, 1);
      colIdx[lrow][5] = getLblNd(i,   j+1, 1);

      colIdx[lrow][6] = getLblNd(i, j-1, 2);
      colIdx[lrow][7] = getLblNd(i, j,   2);
      colIdx[lrow][8] = getLblNd(i, j+1, 2);

      colIdx[lrow][9]  = getLblNd(i-1, j,   3);
      colIdx[lrow][10] = getLblNd(i,   j,   3);
      colIdx[lrow][11] = getLblNd(i-1, j+1, 3);
      colIdx[lrow][12] = getLblNd(i,   j+1, 3);

      colIdx[lrow][13] = getLblNd(i, j,   4);
      colIdx[lrow][14] = getLblNd(i, j+1, 4);
    }
  }

  // Dual volumes: Type 2, on the boundaries parallel to the x-axis

  j = 0;
  for (i=1; i<=nx; ++i) {
    lbl = getLblDualVol(i, j, 2);
    lrow = lbl - 1;
    numEntrsInRow[lrow] = 9;
    colIdx[lrow] = new int[9];
    colIdx[lrow][0] = getLblNd(i-1, j,   1);
    colIdx[lrow][1] = getLblNd(i,   j,   1);
    colIdx[lrow][2] = getLblNd(i-1, j+1, 1);
    colIdx[lrow][3] = getLblNd(i,   j+1, 1);
    colIdx[lrow][4] = getLblNd(i,   j,   2);
    colIdx[lrow][5] = getLblNd(i,   j+1, 2);
    colIdx[lrow][6] = getLblNd(i-1, j+1, 3);
    colIdx[lrow][7] = getLblNd(i,   j+1, 3);
    colIdx[lrow][8] = getLblNd(i,   j+1, 4);
  }

  j = ny;
  for (i=1; i<=nx; ++i) {
    lbl = getLblDualVol(i, j, 2);
    lrow = lbl - 1;
    numEntrsInRow[lrow] = 9;
    colIdx[lrow] = new int[9];
    colIdx[lrow][0] = getLblNd(i-1, j-1, 1);
    colIdx[lrow][1] = getLblNd(i,   j-1, 1);
    colIdx[lrow][2] = getLblNd(i-1, j,   1);
    colIdx[lrow][3] = getLblNd(i,   j,   1);
    colIdx[lrow][4] = getLblNd(i,   j-1, 2);
    colIdx[lrow][5] = getLblNd(i,   j,   2);
    colIdx[lrow][6] = getLblNd(i-1, j,   3);
    colIdx[lrow][7] = getLblNd(i,   j,   3);
    colIdx[lrow][8] = getLblNd(i,   j,   4);
  }

  // Dual volumes: Type 3, not on the boundaries 

  for (j=1; j<=ny; ++j) {
    for (i=1; i<=(nx-1); ++i) {
      lbl = getLblDualVol(i, j, 3);
      lrow = lbl - 1;
      numEntrsInRow[lrow] = 15;
      colIdx[lrow] = new int[15];

      colIdx[lrow][0] = getLblNd(i-1, j-1, 1);
      colIdx[lrow][1] = getLblNd(i,   j-1, 1);
      colIdx[lrow][2] = getLblNd(i+1, j-1, 1);
      colIdx[lrow][3] = getLblNd(i-1, j,   1);
      colIdx[lrow][4] = getLblNd(i,   j,   1);
      colIdx[lrow][5] = getLblNd(i+1, j,   1);

      colIdx[lrow][6] = getLblNd(i,   j-1, 2);
      colIdx[lrow][7] = getLblNd(i+1, j-1, 2);
      colIdx[lrow][8] = getLblNd(i,   j,   2);
      colIdx[lrow][9] = getLblNd(i+1, j,   2);

      colIdx[lrow][10] = getLblNd(i-1, j, 3);
      colIdx[lrow][11] = getLblNd(i,   j, 3);
      colIdx[lrow][12] = getLblNd(i+1, j, 3);

      colIdx[lrow][13] = getLblNd(i,   j, 4);
      colIdx[lrow][14] = getLblNd(i+1, j, 4);
    }
  }

  // Dual volumes: Type 3, on the boundaries parallel to the y-axis

  i = 0;
  for (j=1; j<=ny; ++j) {
    lbl = getLblDualVol(i, j, 3);
    lrow = lbl - 1;
    numEntrsInRow[lrow] = 9;
    colIdx[lrow] = new int[9];
    colIdx[lrow][0] = getLblNd(i,   j-1, 1);
    colIdx[lrow][1] = getLblNd(i+1, j-1, 1);
    colIdx[lrow][2] = getLblNd(i,   j,   1);
    colIdx[lrow][3] = getLblNd(i+1, j,   1);
    colIdx[lrow][4] = getLblNd(i+1, j-1, 2);
    colIdx[lrow][5] = getLblNd(i+1, j,   2);
    colIdx[lrow][6] = getLblNd(i,   j,   3);
    colIdx[lrow][7] = getLblNd(i+1, j,   3);
    colIdx[lrow][8] = getLblNd(i+1, j,   4);
  }

  i = nx;
  for (j=1; j<=ny; ++j) {
    lbl = getLblDualVol(i, j, 3);
    lrow = lbl - 1;
    numEntrsInRow[lrow] = 9;
    colIdx[lrow] = new int[9];
    colIdx[lrow][0] = getLblNd(i-1, j-1, 1);
    colIdx[lrow][1] = getLblNd(i,   j-1, 1);
    colIdx[lrow][2] = getLblNd(i-1, j,   1);
    colIdx[lrow][3] = getLblNd(i,   j,   1);
    colIdx[lrow][4] = getLblNd(i,   j-1, 2);
    colIdx[lrow][5] = getLblNd(i,   j,   2);
    colIdx[lrow][6] = getLblNd(i-1, j,   3);
    colIdx[lrow][7] = getLblNd(i,   j,   3);
    colIdx[lrow][8] = getLblNd(i,   j,   4);
  }

  // Dual volumes: Type 4

  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      lbl = getLblDualVol(i, j, 4);
      lrow = lbl - 1;
      numEntrsInRow[lrow] = 9;
      colIdx[lrow] = new int[9];
      colIdx[lrow][0] = getLblNd(i-1, j-1, 1);
      colIdx[lrow][1] = getLblNd(i,   j-1, 1);
      colIdx[lrow][2] = getLblNd(i-1, j,   1);
      colIdx[lrow][3] = getLblNd(i,   j,   1);
      colIdx[lrow][4] = getLblNd(i,   j-1, 2);
      colIdx[lrow][5] = getLblNd(i,   j,   2);
      colIdx[lrow][6] = getLblNd(i-1, j,   3);
      colIdx[lrow][7] = getLblNd(i,   j,   3);
      colIdx[lrow][8] = getLblNd(i,   j,   4);
    }
  }

  glbMassMat.reshape(numRows, numCols, numEntrsInRow, colIdx);
  glbStfMat.reshape(numRows, numCols, numEntrsInRow, colIdx);

  for (i=0; i<numRows; ++i)  delete[] colIdx[i];
  delete[] colIdx, numEntrsInRow;

  return;
}


// Quadri2Mesh: assembling the global mass matrix

void Quadri2Mesh::asmGlbMassMat(SparseMatrix &glbMassMat) 
{
  int i, ii, j, jj, lblVrtx[10];
  double a;
  FullMatrix eltMassMat(9,9);

  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      eltMassMat = cmptEltMassMat(i, j);

      // lblVrtx[0] not used
      lblVrtx[1] = getLblNd(i-1, j-1, 1);
      lblVrtx[2] = getLblNd(i,   j-1, 1);
      lblVrtx[3] = getLblNd(i,   j,   1);
      lblVrtx[4] = getLblNd(i-1, j,   1);
      lblVrtx[5] = getLblNd(i,   j-1, 2);
      lblVrtx[7] = getLblNd(i,   j,   2);
      lblVrtx[6] = getLblNd(i,   j,   3);
      lblVrtx[8] = getLblNd(i-1, j,   3);
      lblVrtx[9] = getLblNd(i,   j,   4);

      for (ii=1; ii<=9; ++ii) {
        for (jj=1; jj<=9; ++jj) {
          a  = eltMassMat(ii,jj);
          glbMassMat.addEntry(lblVrtx[ii], lblVrtx[jj], a);
        }
      }
    }
  }

  return;
}


// Quadri2Mesh: assembling the global stiffness matrix

void Quadri2Mesh::asmGlbStfMat(SparseMatrix &glbStfMat, double (*perm)(double*)) 
{
  int i, ii, j, jj, lblVrtx[10];
  double a;
  FullMatrix eltStfMat(9,9);

  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      eltStfMat = cmptEltStfMat(i, j, perm);

      // lblVrtx[0] not used
      lblVrtx[1] = getLblNd(i-1, j-1, 1);
      lblVrtx[2] = getLblNd(i,   j-1, 1);
      lblVrtx[3] = getLblNd(i,   j,   1);
      lblVrtx[4] = getLblNd(i-1, j,   1);
      lblVrtx[5] = getLblNd(i,   j-1, 2);
      lblVrtx[7] = getLblNd(i,   j,   2);
      lblVrtx[6] = getLblNd(i,   j,   3);
      lblVrtx[8] = getLblNd(i-1, j,   3);
      lblVrtx[9] = getLblNd(i,   j,   4);

      for (ii=1; ii<=9; ++ii) {
        for (jj=1; jj<=9; ++jj) {
          a  = eltStfMat(ii,jj);
          glbStfMat.addEntry(lblVrtx[ii], lblVrtx[jj], a);
        }
      }
    }
  }

  return;
}


void Quadri2Mesh::asmGlbMats0(SparseMatrix &glbMassMat, 
  SparseMatrix &glbStfMat, double (*perm)(double*)) 
{
  asmGlbMassMat(glbMassMat); 
  asmGlbStfMat(glbStfMat, perm); 
}


// Quadri2Mesh: computing the global "meow" matrix

DiagMatrix Quadri2Mesh::cmptGlbMeowMat() const 
{
  int i, j, k, lblDualVol[10];  // lblDualVol[0] not used
  double a = 0.211324865405187;  // 1/(3+sqrt(3))
  PtVec2d P1, P2, P3, P4;
  PtVec2d P12, P21, P23, P32, P34, P43, P41, P14, PP1, PP2, PP3, PP4;
  DualQuadri DQ[10];  // DQ[0] not used

  DiagMatrix glbMeowMat(getNumDualVols());

  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      P1 = nd[(j-1)*(nx+1)+i-1];
      P2 = nd[(j-1)*(nx+1)+i];
      P3 = nd[j*(nx+1)+i];
      P4 = nd[j*(nx+1)+i-1];

      P12 = (1-a)*P1 + a*P2;
      P21 = (1-a)*P2 + a*P1;

      P23 = (1-a)*P2 + a*P3;
      P32 = (1-a)*P3 + a*P2;

      P34 = (1-a)*P3 + a*P4;
      P43 = (1-a)*P4 + a*P3;

      P41 = (1-a)*P4 + a*P1;
      P14 = (1-a)*P1 + a*P4;

      PP1 = (1-a)*(1-a)*P1 +     a*(1-a)*P2 +         a*a*P3 +     (1-a)*a*P4;
      PP2 =     a*(1-a)*P1 + (1-a)*(1-a)*P2 +     (1-a)*a*P3 +         a*a*P4;
      PP3 =         a*a*P1 +     (1-a)*a*P2 + (1-a)*(1-a)*P3 +     a*(1-a)*P4;
      PP4 =     (1-a)*a*P1 +         a*a*P2 +     a*(1-a)*P3 + (1-a)*(1-a)*P4;

      DQ[1] = DualQuadri(P1,P12,PP1,P14);
      DQ[2] = DualQuadri(P2,P23,PP2,P21);
      DQ[3] = DualQuadri(P3,P34,PP3,P32);
      DQ[4] = DualQuadri(P4,P41,PP4,P43);
      DQ[5] = DualQuadri(P12,P21,PP2,PP1);
      DQ[6] = DualQuadri(P23,P32,PP3,PP2);
      DQ[7] = DualQuadri(P34,P43,PP4,PP3);
      DQ[8] = DualQuadri(P41,P14,PP1,PP4);
      DQ[9] = DualQuadri(PP1,PP2,PP3,PP4);

      lblDualVol[1] = getLblDualVol(i-1, j-1, 1);
      lblDualVol[2] = getLblDualVol(i,   j-1, 1);
      lblDualVol[3] = getLblDualVol(i,   j,   1);
      lblDualVol[4] = getLblDualVol(i-1, j,   1);
      lblDualVol[5] = getLblDualVol(i,   j-1, 2);
      lblDualVol[7] = getLblDualVol(i,   j,   2);
      lblDualVol[6] = getLblDualVol(i,   j,   3);
      lblDualVol[8] = getLblDualVol(i-1, j,   3);
      lblDualVol[9] = getLblDualVol(i,   j,   4);

      for (k=1; k<=9; ++k)  glbMeowMat.addEntry(lblDualVol[k], DQ[k].area());
    }
  }

  return glbMeowMat;
}


// Quadri2Mesh: evaluating a scalar function at all nodes on the mesh 

Vector Quadri2Mesh::evalFxnAllNds(double (*fxn)(double*), double t) const
{
  Vector val((2*nx+1)*(2*ny+1));

  double *txy = new double[3];
  txy[0] = t;
 
  for (int k=0; k<(2*nx+1)*(2*ny+1); ++k) {
    txy[1] = nd[k].xCrd();
    txy[2] = nd[k].yCrd();
    val[k] = fxn(txy);
  }

  delete[] txy;
  return val;
}


// Quadri2Mesh: computing the L_2 norm of the error of 
// the nodal (affine) biquadratic interpolation of a function

double Quadri2Mesh::cmptL2NormErrIntpln(double (*fxn)(double*), 
  double t) const
{
  int i, j, k;
  double intgrl, jac, sum2, X, Y, Z, z, zZ2;
  PtVec2d P1, P2, P3, P4, P;
  Vector val((2*nx+1)*(2*ny+1));

  double *txy = new double[3];
  txy[0] = t;

  for (k=0; k<(2*nx+1)*(2*ny+1); ++k) {
    txy[1] = nd[k].xCrd();
    txy[2] = nd[k].yCrd();
    val[k] = fxn(txy);
  }

  sum2 = 0.0;
  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      P1 = nd[(j-1)*(nx+1)+i-1];
      P2 = nd[(j-1)*(nx+1)+i];
      P3 = nd[j*(nx+1)+i];
      P4 = nd[j*(nx+1)+i-1];

      Quadri2 Q2(P1, P2, P3, P4);
      DualQuadri DQ(P1, P2, P3, P4);

      intgrl = 0.0;
      for (k=0; k<GQR5npts; ++k) {
        X = (GQR5wab[k][1]+1)/2.0;
        Y = (GQR5wab[k][2]+1)/2.0;
		
		Z = 0.0;
        Z += val(getLblNd(i-1,j-1,1))*Q2.locBasisFxn(X,Y,1);
        Z += val(getLblNd(i,j-1,1))*Q2.locBasisFxn(X,Y,2);
        Z += val(getLblNd(i,j,1))*Q2.locBasisFxn(X,Y,3);
        Z += val(getLblNd(i-1,j,1))*Q2.locBasisFxn(X,Y,4);
        Z += val(getLblNd(i,j-1,2))*Q2.locBasisFxn(X,Y,5);
        Z += val(getLblNd(i,j,3))*Q2.locBasisFxn(X,Y,6);
        Z += val(getLblNd(i,j,2))*Q2.locBasisFxn(X,Y,7);
        Z += val(getLblNd(i-1,j,3))*Q2.locBasisFxn(X,Y,8);
        Z += val(getLblNd(i,j,4))*Q2.locBasisFxn(X,Y,9);

        P = Q2.mapping(X,Y);
        txy[1] = P.xCrd();
        txy[2] = P.yCrd();
        z = fxn(txy);

        zZ2 = fabs(z-Z)*fabs(z-Z);
        jac = DQ.jacobian1(GQR5wab[k][1],GQR5wab[k][2]);
        intgrl += zZ2*fabs(jac)*GQR5wab[k][0];
      }
      sum2 += intgrl;
    }
  }

  delete[] txy;
  return sqrt(sum2);
}


// Quadri2Mesh: computing the L2-norm of the error 
// an exact solution vs an approximate solution (e.g. FVE)

double Quadri2Mesh::cmptL2NormErrAprxSln(double (*fxn)(double*), 
  double t, const Vector &U) const 
{
  int i, j, k;
  double intgrl, jac, sum2, X, Y, Z, z, zZ2;
  PtVec2d P1, P2, P3, P4, P;

  double *txy = new double[3];
  txy[0] = t;

  sum2 = 0.0;
  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      P1 = nd[(j-1)*(nx+1)+i-1];
      P2 = nd[(j-1)*(nx+1)+i];
      P3 = nd[j*(nx+1)+i];
      P4 = nd[j*(nx+1)+i-1];

      Quadri2 Q2(P1, P2, P3, P4);
      DualQuadri DQ(P1, P2, P3, P4);

	  intgrl = 0.0;
      for (k=0; k<GQR5npts; ++k) {
        X = (GQR5wab[k][1]+1)/2.0;
        Y = (GQR5wab[k][2]+1)/2.0;
		
        P = Q2.mapping(X,Y);
        txy[1] = P.xCrd();
        txy[2] = P.yCrd();
        z = fxn(txy);

		Z = 0.0;
        Z += U(getLblNd(i-1,j-1,1))*Q2.locBasisFxn(X,Y,1);
        Z += U(getLblNd(i,j-1,1))*Q2.locBasisFxn(X,Y,2);
        Z += U(getLblNd(i,j,1))*Q2.locBasisFxn(X,Y,3);
        Z += U(getLblNd(i-1,j,1))*Q2.locBasisFxn(X,Y,4);
        Z += U(getLblNd(i,j-1,2))*Q2.locBasisFxn(X,Y,5);
        Z += U(getLblNd(i,j,3))*Q2.locBasisFxn(X,Y,6);
        Z += U(getLblNd(i,j,2))*Q2.locBasisFxn(X,Y,7);
        Z += U(getLblNd(i-1,j,3))*Q2.locBasisFxn(X,Y,8);
        Z += U(getLblNd(i,j,4))*Q2.locBasisFxn(X,Y,9);

        zZ2 = fabs(z-Z)*fabs(z-Z);
        jac = DQ.jacobian1(GQR5wab[k][1],GQR5wab[k][2]);
        intgrl += zZ2*fabs(jac)*GQR5wab[k][0];
      }
      sum2 += intgrl;
    }
  }

  delete[] txy;
  return sqrt(sum2);
}


// Quadri2Mesh: computing the L2-norm of the error 
// an exact solution vs an approximate solution (e.g. FVE)

double Quadri2Mesh::cmptH1SemiNormErrAprxSln(double (*fxnux)(double*), 
  double (*fxnuy)(double*), double t, const Vector &U) const 
{
  int i, j, k;
  double intgrl, jac, sum2, X, Y, w;
  PtVec2d P, P1, P2, P3, P4, Ug, ug, vg;

  double *txy = new double[3];
  txy[0] = t;

  sum2 = 0.0;
  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      P1 = nd[(j-1)*(nx+1)+i-1];
      P2 = nd[(j-1)*(nx+1)+i];
      P3 = nd[j*(nx+1)+i];
      P4 = nd[j*(nx+1)+i-1];

      Quadri2 Q2(P1, P2, P3, P4);
      DualQuadri DQ(P1, P2, P3, P4);

	  intgrl = 0.0;
      for (k=0; k<GQR5npts; ++k) {
        X = (GQR5wab[k][1]+1)/2.0;
        Y = (GQR5wab[k][2]+1)/2.0;
		
        P = Q2.mapping(X,Y);
        txy[1] = P.xCrd();
        txy[2] = P.yCrd();
		ug = PtVec2d(fxnux(txy),fxnuy(txy));

		Ug = PtVec2d(0,0);
        Ug = Ug + U(getLblNd(i-1,j-1,1))*Q2.locBasisFxnGrad(X,Y,1);
        Ug = Ug + U(getLblNd(i,j-1,1))*Q2.locBasisFxnGrad(X,Y,2);
        Ug = Ug + U(getLblNd(i,j,1))*Q2.locBasisFxnGrad(X,Y,3);
        Ug = Ug + U(getLblNd(i-1,j,1))*Q2.locBasisFxnGrad(X,Y,4);
        Ug = Ug + U(getLblNd(i,j-1,2))*Q2.locBasisFxnGrad(X,Y,5);
        Ug = Ug + U(getLblNd(i,j,3))*Q2.locBasisFxnGrad(X,Y,6);
        Ug = Ug + U(getLblNd(i,j,2))*Q2.locBasisFxnGrad(X,Y,7);
        Ug = Ug + U(getLblNd(i-1,j,3))*Q2.locBasisFxnGrad(X,Y,8);
        Ug = Ug + U(getLblNd(i,j,4))*Q2.locBasisFxnGrad(X,Y,9);

		vg = ug - Ug;
		w = vg.l2norm();

		jac = DQ.jacobian1(GQR5wab[k][1],GQR5wab[k][2]);
        intgrl += (w*w)*fabs(jac)*GQR5wab[k][0];
      }
      sum2 += intgrl;
	}
  }

  delete[] txy;
  return sqrt(sum2);
}


// Quadri2Mesh: computing the norm0 of 
// a piecewise affine biquadratic function (pwabf)

double Quadri2Mesh::cmptNorm0(const SparseMatrix glbMassMat,
  const Vector &U) const
{
  double sum2 = U*(glbMassMat*U);
  return sqrt(sum2);
}


// Quadri2Mesh: computing the norm0h of 
// a piecewise affine biquadratic function (pwabf)

double Quadri2Mesh::cmptNorm0h(const DiagMatrix glbMeowMat, 
  const Vector &U) const 
{
  double sum2=0.0;
  for (int label=1; label<=getNumDualVols(); ++label) {
    sum2 += glbMeowMat.getEntry(label)*U(label)*U(label);
  }
  return sqrt(sum2);
}


// Quadri2Mesh: computing the norm1h (discrete H1 seminorm) of 
// a piecewise affine biquadratic function (pwabf)

double Quadri2Mesh::cmptNorm1h(const Vector &U) const 
{
  int i, ii, j, jj;
  double sum2, DeltaUx[2][3], DeltaUy[3][2];

  sum2 = 0.0;
  for (j=1; j<=ny; ++j) {
    for (i=1; i<=nx; ++i) {
      DeltaUx[0][0] = U(getLblNd(i,j-1,2)) - U(getLblNd(i-1,j-1,1));
      DeltaUx[1][0] = U(getLblNd(i,j-1,1)) - U(getLblNd(i,j-1,2));
      DeltaUx[0][1] = U(getLblNd(i,j,4)) - U(getLblNd(i-1,j,3));
      DeltaUx[1][1] = U(getLblNd(i,j,3)) - U(getLblNd(i,j,4));
      DeltaUx[0][2] = U(getLblNd(i,j,2)) - U(getLblNd(i-1,j,1));
      DeltaUx[1][2] = U(getLblNd(i,j,1)) - U(getLblNd(i,j,2));

      DeltaUy[0][0] = U(getLblNd(i-1,j,3)) - U(getLblNd(i-1,j-1,1));
      DeltaUy[1][0] = U(getLblNd(i,j,4)) - U(getLblNd(i,j-1,2));
      DeltaUy[2][0] = U(getLblNd(i,j,3)) - U(getLblNd(i,j-1,1));
      DeltaUy[0][1] = U(getLblNd(i-1,j,1)) - U(getLblNd(i-1,j,3));
      DeltaUy[1][1] = U(getLblNd(i,j,2)) - U(getLblNd(i,j,4));
      DeltaUy[2][1] = U(getLblNd(i,j,1)) - U(getLblNd(i,j,3));

      for (ii=0; ii<2; ++ii) 
        for (jj=0; jj<3; ++jj) 
          sum2 += DeltaUx[ii][jj]*DeltaUx[ii][jj];
      for (ii=0; ii<3; ++ii) 
        for (jj=0; jj<2; ++jj) 
          sum2 += DeltaUy[ii][jj]*DeltaUy[ii][jj];
    }
  }

  return sqrt(sum2);
}
