function TriMesh = TZdom_TriMesh_GenUnfmF(...
  xa,ya, xb,yb, xc,yc, xd,yd, nx,ny, status) 
%% Generating a uniform "forward" triangular mesh 
% for a trapezoidal domain with vertices (xa,ya), (xb,yb), (xc,yc), (xd,yd) 
% (in counterclockwise orientation) 
% Input: 
%   xa, ya, xb, yb, xc, yc, xd, yd:  Coordinates of 4 vertices 
%       nx: Number of partitions in the x-direction 
%       ny: Number of partitions in the y-direction 
%   status: 1 or 2 
% Output: 
%   TriMesh: a structure of arrays for primary mesh info 
% Rachel Cali, James Liu, Zhuoran Wang, ColoState; 2007/01--2016/09

%% For status=1
TriMesh.NumNds = (nx+1)*(ny+1);
TriMesh.NumEms = 2*nx*ny;

%% Partitions for [0,1]^2 (the unit square) 
x = linspace(0,1,nx+1);
y = linspace(0,1,ny+1);
[X,Y] = meshgrid(x,y);

%% Transformation from the unit square to the trapezoid 
% Coefficients 
A1 = xa;  A2 = xb - xa;  A3 = xd - xa;  A4 = (xa+xc) - (xb+xd);
B1 = ya;  B2 = yb - ya;  B3 = yd - ya;  B4 = (ya+yc) - (yb+yd);
% transformation 
x = A1 + A2*X + A3*Y + A4*(X.*Y);
y = B1 + B2*X + B3*Y + B4*(X.*Y);

%% Setting the lexicongraphical order for all nodes 
TriMesh.node = [x(:),y(:)];

%% Generating elements 
TriMesh.elem = zeros(TriMesh.NumEms,3);
for i=1:nx
  for j=1:ny
    % Left triangle, counterclockwise, starting at the right angle   
    k = (i-1)*(2*ny)+2*j-1;
    TriMesh.elem(k,1) = (i-1)*(ny+1) + (j+1);
    TriMesh.elem(k,2) = TriMesh.elem(k,1) - 1;
    TriMesh.elem(k,3) = TriMesh.elem(k,1) + (ny+1);
    % Right triangle, counterclockwise, starting at the right angle 
    k = (i-1)*(2*ny)+2*j;
    TriMesh.elem(k,1) = i*(ny+1) + j;
    TriMesh.elem(k,2) = TriMesh.elem(k,1) + 1; 
    TriMesh.elem(k,3) = TriMesh.elem(k,1) - (ny+1);
  end
end

if (status==1)
  TriMesh.flag = 1;
  return;
end

%% Further work for status=2
NumEgsDiag = nx*ny;
NumEgsHori = nx*(ny+1);
NumEgsVert = (nx+1)*ny;
TriMesh.NumEgs = NumEgsDiag + NumEgsHori + NumEgsVert;
% 
% Setting up edges 
TriMesh.edge = zeros(TriMesh.NumEgs,2);
% Diagonal edges 
for i=1:nx
  for j=1:ny
    k = (i-1)*ny+j;
    TriMesh.edge(k,1) = (i-1)*(ny+1)+j;
    TriMesh.edge(k,2) = i*(ny+1)+(j+1);
  end
end
% Horizonatl edges 
for i=1:nx
  for j=0:ny
    k = NumEgsDiag + j*nx+i;
    TriMesh.edge(k,1:2) = [(i-1)*(ny+1)+j+1, i*(ny+1)+j+1];
  end
end
% Vertical edges 
for i=0:nx
  for j=1:ny
    k = NumEgsDiag + NumEgsHori + i*ny+j;
    TriMesh.edge(k,1:2) = [i*(ny+1)+j, i*(ny+1)+j+1];
  end
end
% 
% Setting up element-vs-edges 
% disp('Setting up element-vs-edges...'); 
TriMesh.elem2edge = zeros(TriMesh.NumEms,3);
for i=1:nx
  for j=1:ny
    % Left triangle 
    k = (i-1)*(2*ny) + 2*j-1;
    TriMesh.elem2edge(k,1) = (i-1)*ny+j;
    TriMesh.elem2edge(k,2) = NumEgsDiag + j*nx + i;
    TriMesh.elem2edge(k,3) = NumEgsDiag + NumEgsHori + (i-1)*ny+j;
    % Right triangle 
    k = (i-1)*(2*ny) + 2*j;
    TriMesh.elem2edge(k,1) = (i-1)*ny+j;
    TriMesh.elem2edge(k,2) = NumEgsDiag + (j-1)*nx + i;
    TriMesh.elem2edge(k,3) = NumEgsDiag + NumEgsHori + i*ny+j;
  end
end

% Setting up edge-vs-elements 
% disp('Setting up edge-vs-elements...'); 
TriMesh.edge2elem = zeros(TriMesh.NumEgs,2);
CntEmsEg = zeros(TriMesh.NumEgs,1);
for ie=1:TriMesh.NumEms
  LblEg = TriMesh.elem2edge(ie,1:3);
  CntEmsEg(LblEg) = CntEmsEg(LblEg) + 1;
  for k=1:3
    TriMesh.edge2elem(LblEg(k),CntEmsEg(LblEg(k))) = ie;
  end
end

% Adjusting 
% disp('Adjusting...'); 
for ig=1:TriMesh.NumEgs
  if TriMesh.edge2elem(ig,1)>TriMesh.edge2elem(ig,2)
    tmp = TriMesh.edge2elem(ig,1);
    TriMesh.edge2elem(ig,1) = TriMesh.edge2elem(ig,2);
    TriMesh.edge2elem(ig,2) = tmp;
  end
end
for ig=1:TriMesh.NumEgs
  if TriMesh.edge2elem(ig,1)==0
    TriMesh.edge2elem(ig,1) = TriMesh.edge2elem(ig,2);
    TriMesh.edge2elem(ig,2) = 0;
  end
end

%% Geometric info on element areas and edge lengths 
% elements vs nodes 
k1 = TriMesh.elem(:,1);  k2 = TriMesh.elem(:,2);  k3 = TriMesh.elem(:,3);
x1 = TriMesh.node(k1,1);  y1 = TriMesh.node(k1,2); 
x2 = TriMesh.node(k2,1);  y2 = TriMesh.node(k2,2);
x3 = TriMesh.node(k3,1);  y3 = TriMesh.node(k3,2);
% areas for all elements 
TriMesh.area = 0.5*((x2-x1).*(y3-y1)-(x3-x1).*(y2-y1));
% edges vs nodes 
k1 = TriMesh.edge(:,1);  k2 = TriMesh.edge(:,2);
x1 = TriMesh.node(k1,1);  y1 = TriMesh.node(k1,2);  
x2 = TriMesh.node(k2,1);  y2 = TriMesh.node(k2,2);  
TriMesh.LenEg = sqrt((x2-x1).^2+(y2-y1).^2);

%% Finishing enrichment of secondary mesh info 
TriMesh.flag = 2;

%% Afternotes 
% Mesh is obtained by backslashing!
% Formula for node label 
%   pos=(i,j) i[0,nx],j[0,ny], label=i*(ny+1)+j+1;
% Formula for element label 
%     pos=(i,j): i[1,nx],j[1,ny], 
%    lower/left: label=(i-1)*(2*ny)+2*j-1;
%   upper/right: label=(i-1)*(2*ny)+2*j;
% Formulas for edge label within each group 
%   Diag:  pos=(i,j) i[1,nx],j[1,ny], label=(i-1)*ny+j;
%   Hori:  pos=(i,j) i[1,nx],j[0,ny], label=j*nx+i;
%   Vert:  pos=(i,j) i[0,nx],j[1,ny], label=i*ny+j;

return;