function GM = Hdiv_TriBDM1_EgBas_GramMat(TriMesh)
%% "Hdiv" local BDM1 on a triangle mesh: Gram matrices for the edge-based basis 
% See notes for mathematical simplification 
% James Liu, ColoState; 2012/07--2017/02 

%% Mesh info 
NumEms = TriMesh.NumEms;
% NumEgs = TriMesh.NumEgs;
k1 = TriMesh.elem(:,1);  k2 = TriMesh.elem(:,2);  k3 = TriMesh.elem(:,3);
x1 = TriMesh.node(k1,1);  y1 = TriMesh.node(k1,2); 
x2 = TriMesh.node(k2,1);  y2 = TriMesh.node(k2,2);
x3 = TriMesh.node(k3,1);  y3 = TriMesh.node(k3,2);
area = TriMesh.area;
eg1 = [x3,y3] - [x2,y2];
eg2 = [x1,y1] - [x3,y3];
eg3 = [x2,y2] - [x1,y1];
lg1 = sqrt(eg1(:,1).^2+eg1(:,2).^2);
lg2 = sqrt(eg2(:,1).^2+eg2(:,2).^2);
lg3 = sqrt(eg3(:,1).^2+eg3(:,2).^2);
tan = zeros(NumEms,3,2);
tan(:,1,:) = eg1./[lg1,lg1];
tan(:,2,:) = eg2./[lg2,lg2];
tan(:,3,:) = eg3./[lg3,lg3];
% cosine theta 
ct1 = (lg2.^2+lg3.^2-lg1.^2)./(2*lg2.*lg3);
ct2 = (lg3.^2+lg1.^2-lg2.^2)./(2*lg3.*lg1);
ct3 = (lg1.^2+lg2.^2-lg3.^2)./(2*lg1.*lg2);
% sine theta 
st1 = sqrt(1-ct1.^2);
st2 = sqrt(1-ct2.^2);
st3 = sqrt(1-ct3.^2);
% Dot products among the three edge tangetial vectors for all elements 
dptt = zeros(NumEms,3,3);
for i=1:3
  for j=1:3
    dptt(:,i,j) = dot(tan(:,i,:),tan(:,j,:),3);
  end
end

%% Gram matrices in BDM1 edge-based basis 
% For each element, the Gram matrix is 6x6 
GM = zeros(NumEms,6,6);
% diagonal entries 
GM(:,1,1) = (area/6)./(st2.^2);  GM(:,2,2) = (area/6)./(st3.^2);
GM(:,3,3) = (area/6)./(st3.^2);  GM(:,4,4) = (area/6)./(st1.^2);
GM(:,5,5) = (area/6)./(st1.^2);  GM(:,6,6) = (area/6)./(st2.^2);
% 1st row off-diagonals 
GM(:,1,2) = -dptt(:,3,2)./(st2.*st3).*(area/12);
GM(:,1,3) =  dptt(:,3,1)./(st2.*st3).*(area/12);
GM(:,1,4) = -dptt(:,3,3)./(st2.*st1).*(area/12);
GM(:,1,5) =  dptt(:,3,2)./(st2.*st1).*(area/12);
GM(:,1,6) = -dptt(:,3,1)./(st2.*st2).*(area/6);
% 2nd row off-diagonals 
GM(:,2,3) = -dptt(:,2,1)./(st3.*st3).*(area/6);
GM(:,2,4) =  dptt(:,2,3)./(st3.*st1).*(area/12);
GM(:,2,5) = -dptt(:,2,2)./(st3.*st1).*(area/12);
GM(:,2,6) =  dptt(:,2,1)./(st3.*st2).*(area/12);
% 3rd row off-diagonals 
GM(:,3,4) = -dptt(:,1,3)./(st3.*st1).*(area/12);
GM(:,3,5) =  dptt(:,1,2)./(st3.*st1).*(area/12);
GM(:,3,6) = -dptt(:,1,1)./(st3.*st2).*(area/12);
% 4th row off-diagonals 
GM(:,4,5) = -dptt(:,3,2)./(st1.*st1).*(area/6);
GM(:,4,6) =  dptt(:,3,1)./(st1.*st2).*(area/12);
% 5th row off-diagonals 
GM(:,5,6) = -dptt(:,2,1)./(st1.*st2).*(area/12);
% Symmetry 
for i=2:6 
  for j=1:(i-1) 
    GM(:,i,j) = GM(:,j,i);
  end
end

return;